package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.time.LocalDateTime;

/**
 * Timed task for the loaner to collect money without a prior auction.
 * Created by andi on 26.03.17.
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class LoanerCollectMoneyWithoutAuctionTimedTask extends TimedTask {
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;

    private IDebt debt;
    private IPlayer debitor;

    public LoanerCollectMoneyWithoutAuctionTimedTask(IDebt debt, IPlayer debitor, LocalDateTime deadline) {
        this.debt = debt;
        this.debitor = debitor;
        setExecutionTime(deadline);
    }

    @Override
    public void run() {
        ICompany company = debitor.getCompany();
        int requiredAmount = debt.getAmount();
        int amount = (int) Math.min(requiredAmount, company.getCash());
        if (debitor instanceof IHumanPlayer) {
            company.updateCash(-amount);
        } else {
            company.updateCashDirectly(-amount);
        }
        if (debt.getCreditor() instanceof IPlayer) {
            if (debt.getCreditor() instanceof IHumanPlayer) {
                ((IPlayer) debt.getCreditor()).getCompany().updateCash(amount);
            } else {
                ((IPlayer) debt.getCreditor()).getCompany().updateCashDirectly(amount);
            }
            if (debt.getCreditor() instanceof IHumanPlayer) {
                DisplayMessage msg = new DisplayMessage(EMessageCategory.TRADE, "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.paybackAfterWithoutAuction", new Object[]{debitor.getName(), debitor.getLastName(), amount});
                TargetedEvent message = new TargetedEvent((IHumanPlayer) debt.getCreditor(), msg);
                clientServerEventBus.post(message);
            }
        }
    }

    @Override
    public String toString() {
        return "LoanerCollectMoneyWithoutAuctionTimedTask{" +
                "UUID=" + getUuid() +
                ", execution at " + getExecutionTime() +
                ", debt=" + debt +
                ", debitor=" + debitor.getName() + " " + debitor.getLastName() +
                '}';
    }
}
