package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.game.event.data.PeriodicalDailyUpdate;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.people.IContractBrokers;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.people.ContractBrokerConnections;
import ch.sahits.game.openpatrician.model.people.IContractBroker;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.service.ModelTranslations;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * List of all contract brokers
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Apr 30, 2018
 */
@Component
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
public class ContractBrokers implements IContractBrokers {
    @ListType(ContractBrokerConnections.class)
    private List<ContractBrokerConnections> brokers = new ArrayList<>();

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;

    @Autowired
    @XStreamOmitField
    private ModelTranslations translationService;
    @Autowired
    @XStreamOmitField
    private DialogTemplateFactory dialogTemplateFactory;

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }

    @Override
    public void add(IContractBroker broker, ICity city) {
        long numbers =brokers.stream().filter(connection -> connection.getCity().equals(city)).count();
        Preconditions.checkArgument(numbers == 0, "There can only be on contract broker be assigned to a city at a time.");
        brokers.add(new ContractBrokerConnections(broker, city));
    }

    /**
     * Assign a player to a contract broker.
     * @param player to be assigned
     * @param broker to whom the player is assigned.
     * @param deadline until when the wares have to be delivered.
     */
    public void assignPlayer(IPlayer player, IContractBroker broker, LocalDateTime deadline) {
        brokers.stream()
                .filter(connection -> connection.getContractBroker().equals(broker))
                .forEach(connection -> {
                    connection.setAssignedPlayer(Optional.of(player));
                    connection.setDeadline(deadline);
                });
    }

    /**
     * Remove the connections matching a specific broker.
     * @param broker to be removed.
     */
    public void remove(IContractBroker broker) {
        brokers.removeIf(connection -> connection.getContractBroker().equals(broker));
    }

    /**
     * Daily checking if the deadline has passed.
     * @param event
     */
    @Subscribe
    public void removePastDeadline(PeriodicalDailyUpdate event) {
        for (Iterator<ContractBrokerConnections> iterator = brokers.iterator(); iterator.hasNext(); ) {
            ContractBrokerConnections connection =  iterator.next();
            if (connection.getDeadline() != null && connection.getDeadline().isBefore(event.getDateTime())) {
                 iterator.remove();
                 if (connection.getAssignedPlayer().get() instanceof IHumanPlayer) {
                     IContractBroker broker = connection.getContractBroker();
                     DialogTemplateParameterSupplier parameterSupplier =
                             new DialogTemplateParameterSupplier(new Object[]{broker.getAmountAndPrice().getAmount(),
                                     translationService.getLocalDisplayName((EWare) broker.getWare()), broker.getName(),
                                     connection.getCity().getName(),
                                     translationService.toDisplayString(connection.getDeadline())});
                     DialogTemplate template = dialogTemplateFactory.createDialogTemplate(EDialogTemplateType.CONTRACT_BROKER_NO_DELIVERY, parameterSupplier);
                     DisplayTemplateMessage message = new DisplayTemplateMessage(EMessageCategory.PERSONAL, "ch.sahits.game.openpatrician.engine.land.city.ContractBrokers.unreliable", template);
                     TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer) connection.getAssignedPlayer().get(), message);
                     clientServerEventBus.post(tagetDisplayMsg);
                 }
            }
        }
    }

    @Override
    public Optional<IContractBroker> findContractBroker(ICity city, IPlayer player) {
        return brokers.stream().filter(connection ->
            connection.getCity().equals(city)
                    && connection.getAssignedPlayer().isPresent()
                    && connection.getAssignedPlayer().get().equals(player))
                .map(connection -> connection.getContractBroker()).findFirst();
    }

    @Override
    public boolean hasContractBroker(ICity city) {
        return brokers.stream().filter(connection -> connection.getCity().equals(city)).findFirst().isPresent();
    }
}
