package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedCheckedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesSellTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AmountBasedAggregatedDumpTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.BuyWeaponTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckForRepairTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckHireCaptainTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.GuildJoinTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireDismissTradeManagerTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireSailorsStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.PayBackLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TakeLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TravelToTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.UpgradeShipTradeStep;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.util.Pair;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * This trade strategy focuses on the cheap production of wares and sells them in
 * the nearest city where there is a need.
 */
@Slf4j
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class CheapProductionTradeStrategy extends BasePlayerTradeStrategy {

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;
//    @Value("${aiplayer.money.low}")
//    private int lowMoney = 2000;
    @Autowired
    @XStreamOmitField
    private MapService mapService;

    public CheapProductionTradeStrategy() {
        this.tradeStrategyType = EAITradeStrategyType.CHEAP_PRODUCTION;
    }

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
    }

    @Override
    public void initializeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        // get current city
        Optional<ICity> optCity = shipService.findCity(vessel);
        if (!optCity.isPresent()) {
            log.debug("The vessel {} ({}) is not in any city but {}", vessel.getName(), vessel.getUuid(), vessel.getLocation());
        }
        // sell all loaded wares
        List<IWare> loadedWares = getLoadedWares(vessel);
        ICity city = optCity.get();
        if (!loadedWares.isEmpty()) {
            AggregatesSellTradeStep sellStep = createAggregatedSellStep(vessel, city, loadedWares);
            append(player, vessel, sellStep);
            int maxAmount = vessel.getSize() / 5;
            AmountBasedAggregatedDumpTradeStep sellStep1 = createConditionalAggregatedDumpStep(vessel, city, loadedWares, maxAmount);
            append(player, vessel, sellStep1);
        }

        PayBackLoanTradeStep payBackLoan = createPaybackLoanStep(player, city);
        append(player, vessel, payBackLoan);
        CheckHireCaptainTradeStep hireCaptain = createHireCaptain(vessel, city, player);
        append(player, vessel, hireCaptain);
        GuildJoinTradeStep joinTradeStep = createJoinGuildTradeStep(vessel, city, player);
        append(player, vessel, joinTradeStep);
        HireDismissTradeManagerTradeStep hireTradeManager = createHireDismissTradeManagerTradeStep(city, player);
        append(player, vessel, hireTradeManager);
        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, city);
        append(player, vessel, takeLoanStep);
        BuyWeaponTradeStep buyWeaponsStep = createWeaponBuyTradeStep(vessel, player, city);
        append(player, vessel, buyWeaponsStep);

        // check wares that are produced cheap and are available
        Optional<IWare> ware = findWareToBuy(city, player);
        if (!ware.isPresent()) {
            // Check for repair
            moveToNextStopEmpty(player, vessel, city);
        } else {
            List<IWare> buyWare = new ArrayList<>();
            buyWare.add(ware.get());
            Optional<ICity> nextStop = findNearestCityWithNeed(city, player, ware.get(), vessel, city);
            if (nextStop.isPresent()) {
                AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, city, buyWare);
                append(player, vessel, buyStepHometown);
                UpgradeShipTradeStep upgradeShipStep = createUpgradeShipTradeStep(vessel, player);
                append(player, vessel, upgradeShipStep);
                CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
                append(player, vessel, repairStep);
                HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
                append(player, vessel, hireSailors);
                TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop.get());
                append(player, vessel, travelTo);
            } else {
                moveToNextStopEmpty(player, vessel, city);
            }
        }
    }

    private void moveToNextStopEmpty(IAIPlayer player, INavigableVessel vessel, ICity city) {
        UpgradeShipTradeStep upgradeShipStep = createUpgradeShipTradeStep(vessel, player);
        append(player, vessel, upgradeShipStep);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
        append(player, vessel, repairStep);
        ICity nextStop = mapService.findNearestNonBlockadedCity(city.getCoordinates());
        HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
    }

    private Optional<ICity> findNearestCityWithNeed(ICity city, IAIPlayer player, IWare ware, INavigableVessel vessel, ICity excludeCity) {
        IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        List<ICity> cityList = mapService.getNonBlockadedCitiesOrderedByDistance(city, vessel);
        cityList.remove(excludeCity);
        for (ICity iCity : cityList) {
            ICityProductionConsumptionKnowledge cityProduction = knowledge.getKnowlege(iCity);
            List<Pair<IWare, Number>> mostNeeded = getMostNeededWares(cityProduction);
            for (Pair<IWare, Number> pair : mostNeeded) {
                if (pair.getKey().equals(ware) && pair.getValue().intValue() < 10) {
                    return Optional.of(iCity);
                }
            }
        }
        return Optional.empty();
    }

    private Optional<IWare> findWareToBuy(ICity city, IAIPlayer player) {
        IWare[] wares = city.getEffectiveProduction();
        IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        ICityProductionConsumptionKnowledge knowledgeCurrentTown = knowledge.getKnowlege(city);
        IWare ware = null;
        int amount = 0;
        for (IWare good : wares) {
            int storedAmount = knowledgeCurrentTown.getStoredAmount(good);
            if (storedAmount > 0 && knowledgeCurrentTown.getProductionAmount(good) > knowledgeCurrentTown.getConsumptionAmount(good)) {
                if (storedAmount > amount) {
                    ware = good;
                    amount = storedAmount;
                }
            }
        }
        return Optional.ofNullable(ware);
    }

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        // no move steps defined
        INavigableVessel vessel = event.getShip();
        if (isMatchingTradeStrategy(vessel)) {
            IAIPlayer player = (IAIPlayer) vessel.getOwner();
            if (hasMoreTradeSteps(player, vessel)) {
                StringBuilder sb = new StringBuilder();
                sb.append("Remaining trade steps when there should be none for ").append(vessel.getName())
                        .append(" (").append(vessel.getUuid()).append(") of ")
                        .append(player.getName()).append(" ").append(player.getLastName()).append(":");
                for (ITradeStep tradeStep : player.getTradeSteps(vessel)) {
                    sb.append(tradeStep).append(" ");
                }
                throw new IllegalStateException(sb.toString());
            }
            ICity city = event.getCity();
            Preconditions.checkArgument(city.getCoordinates().equals(vessel.getLocation()), "The vessel " + vessel.getName() + " ("+vessel.getUuid()+") actually is not in city " + city.getName());
            initializeTradeCycle(player, vessel);
            executeTradeSteps(player, vessel);
        }
    }
}
