package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.model.WeaponSlotCount;
import ch.sahits.game.openpatrician.clientserverinterface.service.ConvoyService;
import ch.sahits.game.openpatrician.clientserverinterface.service.OutriggerService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.engine.event.task.WaitForShipArrival;
import ch.sahits.game.openpatrician.engine.sea.DangerService;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.engine.sea.model.PirateActivity;
import ch.sahits.game.openpatrician.engine.sea.model.PirateActivityEntry;
import ch.sahits.game.openpatrician.clientserverinterface.model.WaitingTradeMissionWrapper;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.ship.ConvoyList;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.ISnaikka;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Trade step to decide if a vessel should form a convoy, join a convoy and doing so.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Sep 18, 2018
 */
@ClassCategory({EClassCategory.SINGLETON_BEAN})
@LazySingleton
public class FormOrJoinConvoyStrategy {

    @Autowired
    private DangerService dangerService;
    @Autowired
    private PirateActivity pirateActivity;
    @Autowired
    private ConvoyList convoyList;
    @Autowired
    private ShipService shipService;
    @Autowired
    private OutriggerService outriggerService;
    @Autowired
    private ConvoyService convoyService;
    @Autowired
    private ServerSideTaskFactory taskFactory;
    @Autowired
    private SeafaringService seafaringService;

    private boolean isDangerous() {
        List<PirateActivityEntry> attacks = pirateActivity.getPirateActivity();
        int numberOfAttacks = dangerService.getNumberOfSuccessfulPirateAttacks(attacks);
        int limitNumberOfFewAttacks = pirateActivity.getObservationPeriodInDays() / 14; // one attack every 2 weeks
        return numberOfAttacks > limitNumberOfFewAttacks;
    }

    private boolean hasSmallFleet(IAIPlayer player) {
        return getShipCount(player) < 5;
    }

    private long getShipCount(IAIPlayer player) {
        return player.getFleet().stream()
                // ships not in onvoys or the orleg ship
                .filter(ship -> ship.parentShipProperty().get() == null || ship.equals(ship.parentShipProperty().get()))
                .count();
    }

    private boolean hasMediumFleet(IAIPlayer player) {
        long shipCount = getShipCount(player);
        return shipCount >= 5 && shipCount <= 20;
    }

    private boolean hasLargeFleet(IAIPlayer player) {
        long shipCount = getShipCount(player);
        return shipCount > 20;
    }

    /**
     * Find a convoy of the owner that does not cope with demand (low on free space).
     */
    private Optional<IConvoy> getNonCopingConvoy(IAIPlayer player) {
        for (Iterator<IConvoy> iterator = convoyList.iterator(); iterator.hasNext(); ) {
            IConvoy convoy = iterator.next();
            if (convoy.getOwner().equals(player)) {
                int capacity = convoy.getLoadableSpace();
                int loadedSpace = convoy.getLoadBinding().get();
                int freeSpace = capacity - loadedSpace;
                if (capacity * 0.1 <= freeSpace) { // 90% full
                    return Optional.of(convoy);
                }
            }
        }
        return Optional.empty();
    }

    /**
     * Find a ship of the player that has a strategy and does not cope with demand (low on free space).
     */
    private Optional<INavigableVessel> findStrategyNotCoping(IAIPlayer player, IShip vessel) {
        for (IShip ship : player.getFleet()) {
            if (!ship.equals(vessel)) {
                IAITradeStrategyType strategy = player.getTradeStrategyType(ship);
                if (strategy != null) {
                    int capacity = ship.getLoadableSpace();
                    int loadedSpace = ship.getLoadBinding().get();
                    int freeSpace = capacity - loadedSpace;
                    if (capacity * 0.1 <= freeSpace) {
                        return Optional.of(ship);
                    }
                }
            }
        }
        return Optional.empty();
    }

    private boolean canBeFittedToOrleg(IShip vessel, ICity city) {
        if (vessel instanceof ISnaikka) {
            return false;
        }
        int requiredStrength = outriggerService.getRequiredWeaponStrength(city);
        WeaponSlotCount weaponSlotCount = shipService.getWeaponSlotCount(vessel.getWeaponSlots());
        int weaponSlots = weaponSlotCount.getNbLargeSlots() + 2 * weaponSlotCount.getNbLargeSlots();
        if (requiredStrength <= weaponSlots) {
            // vessel can become Orleg ship
            int sailorOccupiedSpace = 10;
            int upgradeReduction = vessel.getUpgradeSpaceReduction();
            int actualStrength = shipService.calculateShipsWeaponsStrength(vessel);
            int weaponSpaceRequired = vessel.getOccupiedSpaceByWeapons() + Math.max((requiredStrength - actualStrength), actualStrength);
            int targetOccupiedSpace = sailorOccupiedSpace + upgradeReduction + weaponSpaceRequired;
            if (targetOccupiedSpace < vessel.getSize() / 2) {
                 return true;
            }
        }
        return false;
    }

    private Optional<IConvoy> isConvoyInCity(ICity city) {
        // in city or arriving
        for (Iterator<IConvoy> iterator = convoyList.iterator(); iterator.hasNext(); ) {
            IConvoy convoy = iterator.next();
            Optional<ICity> destination = shipService.getDestinationCity(convoy);
            if (destination.isPresent() && destination.get().equals(city)) {
                return Optional.of(convoy);
            }
        }
        return Optional.empty();
    }

    /**
     * check if the vessel needs to be fitted to become an orleg ship.
     * @param vessel to be checked
     * @return true if the ship needs to be fitted as orleg ship.
     */
    public boolean mustBeFittableForOrleg(INavigableVessel vessel) {
        if (vessel instanceof IShip) {
            return !findVesselWithNotCopingStrategy((IShip) vessel, (IAIPlayer) vessel.getOwner()).isPresent();
        } else {
            return false;
        }
    }

    /**
     * Determine if the vessel should form a convoy or join a convoy own or forign and do so.
     * @param vessel for which to check the convoy
     * @param city current city of the vessel.
     */
    public Optional<IConvoy> joinOrFormConvoy(IShip vessel, ICity city) {
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        Optional<? extends INavigableVessel> strategyNotCoping = findVesselWithNotCopingStrategy(vessel, player);
        Optional<IConvoy> convoyInCity = isConvoyInCity(city);
        IAIPlayer owner = (IAIPlayer) vessel.getOwner();
        if (strategyNotCoping.isPresent()) {
            // set to that strategy and original ship joins convoy private
            ICity convoyNextDestination = shipService.getDestinationCity(strategyNotCoping.get()).get();
            boolean waitingShipAtDestination = convoyNextDestination.equals(city);
            boolean waitedOnShipInCity = strategyNotCoping.get().getLocation().equals(city.getCoordinates());
            if (strategyNotCoping.get() instanceof IConvoy) {
                // join that convoy
                IConvoy convoy = (IConvoy) strategyNotCoping.get();
                if (waitedOnShipInCity) {
                    convoyService.join(convoy, vessel);
                } else {
                    // ensure that the convoy waits until joined
                    Runnable completionAction = () -> {
                        convoyService.join(convoy, vessel);
                        owner.setTradeMission(vessel, null);
                        ITradeMissionData blockingConvoyMission = player.getTradeMission(convoy);
                        if (blockingConvoyMission instanceof WaitingTradeMissionWrapper) {
                            player.setTradeMission(convoy,
                                    ((WaitingTradeMissionWrapper) blockingConvoyMission).getTradeMissionData());
                        }
                    };
                    WaitForShipArrival task = taskFactory.waitForShipArrival(vessel, convoy, convoyNextDestination,
                            completionAction, waitingShipAtDestination);
                    owner.setTradeMission(vessel, task);
                    if (!waitingShipAtDestination) {
                        seafaringService.travelBetweenCities(vessel, convoyNextDestination);
                    }
                    WaitingTradeMissionWrapper blockingMission =
                            new WaitingTradeMissionWrapper(player.getTradeMission(convoy));
                    player.setTradeMission(convoy, blockingMission);
                }
                return Optional.of(convoy);
            } else {
                // form private convoy and let the other ship join
                final IConvoy convoy = convoyService.create(vessel, false);
                IShip ship = (IShip) strategyNotCoping.get();
                Runnable completionAction = () -> {
                    convoyService.join(convoy, ship);
                    ITradeMissionData blockingConvoyMission = player.getTradeMission(ship);
                    if (blockingConvoyMission instanceof WaitingTradeMissionWrapper) {
                        player.setTradeMission(convoy,
                                ((WaitingTradeMissionWrapper) blockingConvoyMission).getTradeMissionData());
                    }
                    owner.setTradeMission(ship, null);
                };
                WaitForShipArrival task = taskFactory.waitForShipArrival(convoy, ship, convoyNextDestination,
                        completionAction, waitingShipAtDestination);
                owner.setTradeMission(convoy, task);
                owner.setTradeStrategyType(convoy, owner.getTradeStrategyType(ship));
                if (!city.equals(convoyNextDestination)) {
                    seafaringService.travelBetweenCities(convoy, convoyNextDestination);
                }
                // Other ship should only arrive
                WaitingTradeMissionWrapper blockingMission =
                        new WaitingTradeMissionWrapper(player.getTradeMission(ship));
                player.setTradeMission(strategyNotCoping.get(), blockingMission);
                return Optional.of(convoy);
            }
        } else if ((hasMediumFleet(player) && !hasSmallFleet(player) || hasLargeFleet(player))
                && canBeFittedToOrleg(vessel, city)) {
            // form public convoy
            boolean publicConvoy = !hasLargeFleet(player);
            return Optional.of(convoyService.create(vessel, publicConvoy));
        } else if (convoyInCity.isPresent()) {
            // join convoy
            IConvoy convoy = convoyInCity.get();
            if (convoy.getLocation().equals(city.getCoordinates())) {
                // join
                convoyService.join(convoy, vessel);
            } else {
                Runnable completionAction = () -> {
                    convoyService.join(convoy, vessel);
                    owner.setTradeMission(vessel, null);
                };
                WaitForShipArrival task = taskFactory.waitForShipArrival(vessel, convoy, city,
                        completionAction, true);
                owner.setTradeMission(vessel, task);
            }
            return Optional.of(convoy);
        }
        return Optional.empty();
    }

    private Optional<? extends INavigableVessel> findVesselWithNotCopingStrategy(IShip vessel, IAIPlayer player) {
        Optional<? extends INavigableVessel> strategyNotCoping = getNonCopingConvoy(player);
        if (!strategyNotCoping.isPresent()) {
            strategyNotCoping = findStrategyNotCoping(player, vessel);
        }
        return strategyNotCoping;
    }

    /**
     * Check if there is a need for a ship of the player to join a convoy or form one.
     * @param player for whom to check
     * @return false if there is no need for a ship to be in a convoy.
     */
    public boolean shouldJoinOrFormConvoy(IAIPlayer player) {
        return isDangerous() && hasSmallFleet(player) || getNonCopingConvoy(player).isPresent();
    }
}
