package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.player.TradeRouteMissionData;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedCheckedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesDumpTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesSellTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.BuyWeaponTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckForRepairTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckHireCaptainTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.GuildJoinTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireDismissTradeManagerTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireSailorsStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.PayBackLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TakeLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TravelToTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.UpgradeShipTradeStep;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.map.ITradeRoute;
import ch.sahits.game.openpatrician.model.map.ITradeRouteStop;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.eventbus.AsyncEventBus;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

/**
 * Trade strategy that is based on a preset trade route
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jun 18, 2016
 */
@Slf4j
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class TradeRouteTradeStrategy extends BasePlayerTradeStrategy {

    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private ShipService shipService;
//    @Value("${aiplayer.money.low}")
//    private int lowMoney = 2000;
    @Autowired
    private MapService mapService;

    public TradeRouteTradeStrategy() {
        this.tradeStrategyType = EAITradeStrategyType.TRADE_ROUTE;
    }
    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
    }

    @Override
    public void initializeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        // At the end of the cycle clear and chose a new route.
        Optional<ICity> optCity = shipService.findCity(vessel);
        if (!optCity.isPresent()) {
            log.debug("The vessel {} ({}) is not in any city but {}", vessel.getName(), vessel.getUuid(), vessel.getLocation());
        }
        ICity city = optCity.get();
        ITradeMissionData missionData = player.getTradeMission(vessel);
        if (missionData == null || !(missionData instanceof TradeRouteMissionData)) {
            missionData = selectTradeRoute(city);
            player.setTradeMission(vessel, missionData);
        }
        ITradeRoute route = ((TradeRouteMissionData)missionData).getTradeRoute();
        // sell all loaded wares
        List<IWare> loadedWares = getLoadedWares(vessel);
        if (!loadedWares.isEmpty()) {
            ICity firstCity = ((TradeRouteMissionData) missionData).getTradeRoute().getTradeStops().iterator().next().getTradeStop();
            if (firstCity.equals(city)){
                AggregatesDumpTradeStep dumpStep = createAggregatedDumpStep(vessel, city, loadedWares);
                append(player, vessel, dumpStep);
            } else {
                AggregatesSellTradeStep sellStep = createAggregatedSellStep(vessel, city, loadedWares);
                append(player, vessel, sellStep);
            }
        }

        PayBackLoanTradeStep payBackLoan = createPaybackLoanStep(player, city);
        append(player, vessel, payBackLoan);
        CheckHireCaptainTradeStep hireCaptain = createHireCaptain(vessel, city, player);
        append(player, vessel, hireCaptain);
        GuildJoinTradeStep joinTradeStep = createJoinGuildTradeStep(vessel, city, player);
        append(player, vessel, joinTradeStep);
        HireDismissTradeManagerTradeStep hireTradeManager = createHireDismissTradeManagerTradeStep(city, player);
        append(player, vessel, hireTradeManager);
        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, city);
        append(player, vessel, takeLoanStep);
        BuyWeaponTradeStep buyWeaponsStep = createWeaponBuyTradeStep(vessel, player, city);
        append(player, vessel, buyWeaponsStep);

        ITradeRouteStop stop = null;
        ICity nextStop = null;
        for (Iterator<ITradeRouteStop> iterator = route.getTradeStops().iterator(); iterator.hasNext(); ) {
            ITradeRouteStop routeStop =  iterator.next();
            if (routeStop.getTradeStop().equals(city)){
                stop = routeStop;
                if (iterator.hasNext()) {
                    routeStop =  iterator.next();
                    nextStop = routeStop.getTradeStop();
                } else {
                    nextStop = route.getTradeStops().iterator().next().getTradeStop(); // first
                }
                break;
            }
        }
        List<IWare> buyWares = new ArrayList<>(stop.getWaresToBuy());
        AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, city, buyWares);
        append(player, vessel, buyStepHometown);
        UpgradeShipTradeStep upgradeShipStep = createUpgradeShipTradeStep(vessel, player);
        append(player, vessel, upgradeShipStep);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
        append(player, vessel, repairStep);
        HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
    }

    /**
     * {@inheritDoc}<br>
     * Choose the nearest city from the set defined in the trade route.
     * @param vessel that should travel to a city, so that the trade cycle can be reinitialized
     * @return
     */
    @Override
    public ICity getCityToRestartTradeCycle(INavigableVessel vessel) {
        ITradeMissionData missionData = ((IAIPlayer)vessel.getOwner()).getTradeMission(vessel);
        ITradeRoute route = ((TradeRouteMissionData)missionData).getTradeRoute();
        double distance = Double.MAX_VALUE;
        ICity destination = null;
        for (ITradeRouteStop stop : route.getTradeStops()) {
            double d = vessel.getLocation().distance(stop.getTradeStop().getCoordinates());
            if (d < distance) {
                distance = d;
                destination = stop.getTradeStop();
            }
        }
        return destination;
    }

    private TradeRouteMissionData selectTradeRoute(ICity city) {
        List<ITradeRoute> routes = mapService.findTradeRoutesFirstStop(city);
        if (routes.isEmpty()) {
            routes = mapService.findTradeRoutesSecondStop(city);
        }
        Collections.shuffle(routes);
        ITradeRoute route = routes.get(0);
        ITradeMissionData missionData = new TradeRouteMissionData();
        ((TradeRouteMissionData)missionData).setTradeRoute(route);
        return ((TradeRouteMissionData)missionData);
    }

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        if (isMatchingTradeStrategy(vessel)) {
            IAIPlayer player = (IAIPlayer) vessel.getOwner();
            initializeTradeCycle(player, vessel);
            executeTradeSteps(player, vessel);
        }
    }

}
