package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.event.data.PeriodicalTimeWeekEndUpdate;
import ch.sahits.game.graphic.image.IMapImageServiceFacade;
import ch.sahits.game.openpatrician.clientserverinterface.model.WeaponSlotCount;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.PeopleFactory;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.land.city.ShipyardEngine;
import ch.sahits.game.openpatrician.event.data.GraphInitialisationComplete;
import ch.sahits.game.openpatrician.event.data.RepairFinishedEvent;
import ch.sahits.game.openpatrician.event.data.ShipArrivesAtDestinationEvent;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.NewPirateEvent;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.people.INonFreeSeaPirate;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.people.impl.SeaPiratesState;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IShipGroup;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.service.RandomNameLoader;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import javafx.geometry.Point2D;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.TimerTask;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Engine governing all pirates.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 29, 2013
 *
 */
@Slf4j
@Lazy
@Service
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@DependentInitialisation(StartNewGameBean.class)
public class PirateEngine extends AbstractEngine {
    @Autowired
    private Random rnd;
	@Autowired
	private SeaPiratesState piratesState;
	@Autowired
	private LocationTracker locationTracker;
	@Autowired
	@Qualifier("serverClientEventBus")
	private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    @Qualifier("syncServerClientEventBus")
    private EventBus syncServerClientEventBus;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private IMapImageServiceFacade mapService;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private ShipyardEngine shipyardEngine;
    @Autowired
    private MapService cityService;
    @Autowired
    private Date date;
    @Autowired
    private SeaFightService seaFightService;
    @Autowired
    private PeopleFactory peopleFactory;
    @Autowired
    private AStarHeuristicProvider aStarHeuristicProvider;
    @Autowired
    @Qualifier("schedulableServerThreadPool")
    private ScheduledExecutorService schedulableServerThreadPool;
    @Autowired
    private ShipService shipService;

    private boolean graphInitialized = false;

    private static RandomNameLoader firstNameLoader = new RandomNameLoader("pirate_firstnames.properties");
    private static RandomNameLoader lastNameLoader = new RandomNameLoader("pirate_lastnames.properties");
    private static RandomNameLoader shipLoader = new RandomNameLoader("shipnames.properties");
	@PostConstruct
	private void initialize() {

        clientServerEventBus.register(this);
        timerEventBus.register(this);
        syncServerClientEventBus.register(this);
        // Delay the rest of the initialisation as this posts events that are handled by this engine and must be initialized first.
        TimerTask task = new TimerTask() {
            @Override
            public void run() {
                while (!graphInitialized) {
                    Thread.yield();
                }
                String firstName = firstNameLoader.getRandomName();
                String lastName  = lastNameLoader.getRandomName();
                IShip ship = createInitialPiratesShip();
                ISeaPirate pirate = peopleFactory.createNewPirate(firstName, lastName, ship);
                ship.setOwner(pirate);
                piratesState.add(pirate); // this will trigger a NewPirateEvent
            }
        };
        schedulableServerThreadPool.schedule(task, 10, TimeUnit.SECONDS);
    }
	@PreDestroy
	private void destroy() {
		clientServerEventBus.unregister(this);
		timerEventBus.unregister(this);
        syncServerClientEventBus.unregister(this);
	}
	@Override
	public List<AbstractEngine> getChildren() {
		return Collections.emptyList();
	}
    @Subscribe
    public void initializedGraph(GraphInitialisationComplete event) {
        graphInitialized = true;
    }

	/**
	 * Handle the even when a new pirate becomes available.
	 * @param pirateEvent new pirate emerged
     */
	@Subscribe
	public void handleNewPirate(NewPirateEvent pirateEvent) {
        IShip ship = (IShip) pirateEvent.getPirate().getShip();
        Point2D adjustedSource = aStarHeuristicProvider.findClosest(ship.getLocation());
        ship.setLocation(adjustedSource);
        selectDestination(pirateEvent.getPirate());
	}

    /**
     * Ship has reached it's destination, select a new one.
     * @param event ship arrives at destination
     */
	@Subscribe
	public void handleShipReachesDestination(ShipArrivesAtDestinationEvent event) {
        INavigableVessel vessel = event.getShip();
        if (handlePirateShip(vessel))  {
          int damageInv = vessel.getDamage();
          if (damageInv < 20) {
              selectDestination((ISeaPirate) vessel.getOwner());
          } else {
              sendForRepairs(vessel);
          }
	  }
	}	/**
	 * Select a new destination for the pirate.
	 * @param pirate free sea pirate
     */
	private void selectDestination(ISeaPirate pirate) {
        Point2D destination;
        Point2D shipPosition = pirate.getShip().getLocation();
        if (pirate instanceof INonFreeSeaPirate && ((INonFreeSeaPirate) pirate).roamingNearCity().isPresent()) {
            ICity city = ((INonFreeSeaPirate) pirate).roamingNearCity().get();
            destination = mapService.getRandomPointAtSeaNear(city.getCoordinates());
        } else {
            destination = mapService.getLocationAtOtherEndOfMap(shipPosition);
        }
        // make sure that the source of the pirate location is in the graph
        Point2D adjustedDestination = aStarHeuristicProvider.findClosest(destination);
        log.info("Adjust destination {} -> {}", destination, adjustedDestination);
        aStarHeuristicProvider.addTargetNodeToHeuristic(adjustedDestination);

        log.debug("Source point of pirate is {}, destination {}", shipPosition, destination);
        try {
            Preconditions.checkArgument(mapService.isOnSea((int) shipPosition.getX(), (int) shipPosition.getY()), "Start position of the ship is not on sea: " + shipPosition);
            Preconditions.checkArgument(mapService.isOnSea((int) adjustedDestination.getX(), (int) adjustedDestination.getY()), "The destination is not at sea: " + adjustedDestination);
        } catch (IllegalArgumentException e) {
            throw e;
        }

        seafaringService.travelNotBetweenCities(pirate.getShip(), adjustedDestination);
	}


    /**
     * Check if there is a ship near the new position and decide if an attack should be initiated.
     * @param event ship changes position
     */
	@Subscribe
	public void handleShipPositionUpdate(ShipPositionUpdateEvent event) {
        INavigableVessel vessel = event.getShip();
        if (handlePirateShip(vessel))  {
            // Check if a ship is visible
            List<INavigableVessel> visibleVessels = locationTracker.getShipsInSegments(event.getToLocation(), 50); // TODO: andi 4/28/16 make this a configurable constant 
            if (!visibleVessels.isEmpty()) {
                // Decide on attack
                int nbPirateShips = 1;
                if (vessel instanceof IShipGroup) {
                    nbPirateShips = ((IShipGroup) vessel).getShips().size();
                }
                for (INavigableVessel visibleVessel : visibleVessels) {
                    IShipOwner vissibleOwner = visibleVessel.getOwner();
                    if (!(vissibleOwner instanceof ISeaPirate)) {
                       int nbOpponentShips = 1;
                       if (visibleVessel instanceof IConvoy) {
                           nbOpponentShips = ((IConvoy) visibleVessel).getShips().size();
                       }
                       if (nbOpponentShips <= nbPirateShips) {
                           // Attack
                           seaFightService.calculateOutcome(vessel, visibleVessel, new SeaFightContext(ESeaFightType.STANDARD));
                           if (vissibleOwner instanceof IPlayer) {
                               // TODO: andi 12/17/16 only update if vessel is defeated and not if the fight has been survived 
                               IPlayer owner = (IPlayer) vissibleOwner;
                               owner.updateCrimialDrive(-1);
                           }
                           break;
                       }
                   }
                }
            }
		}
	}

    /**
     * Send a vessel to be repaired.
     * @param vessel to be repaired
     */
	private void sendForRepairs(INavigableVessel vessel) {
		vessel.togglePirateFlag();
        ICity city = cityService.findNearestCity(vessel.getLocation());
        seafaringService.travelToCity(vessel, city);
	}

    /**
     * When a pirate ship enters a port it does so for repair.
     */
    @Subscribe
    public void handleRepairPirateShip(ShipEntersPortEvent event) {
        if (handlePirateShip(event.getShip()))  {
            // repair the ship or group
            INavigableVessel vessel = event.getShip();
            ICity city = event.getCity();
            IShipyard shipyard = city.getCityState().getShipyardState();
            if (vessel instanceof IShip) {
                shipyardEngine.repairPirateShip(shipyard, (IShip) vessel);
            } else if (vessel instanceof IShipGroup) {
                shipyardEngine.repair(shipyard, (IShipGroup)vessel);
            } else {
                log.warn("Can only repair pirate ships and pirate ship groups.");
            }
        }
    }

    /**
     * Handle the event for the finished repair.
     * @param event repair has finished
     */
    @Subscribe
    public void handleRepairFinished(RepairFinishedEvent event) {
        if (handlePirateShip(event.getShip())) {
            INavigableVessel vessel = event.getShip();
            Point2D destination = mapService.getRandomPointAtSeaNear(vessel.getLocation());
            seafaringService.travelFromCity(vessel, destination);
            vessel.togglePirateFlag();
        }

    }

    /**
     * Periodically check if there is a free pirate and if not, create one.
     * @param event weekly update
     */
    @Subscribe
    public void handleWeeklyUpdate(PeriodicalTimeWeekEndUpdate event) {
        // Check if there is a free pirate and if not spawn a new one
        if (piratesState.getFreePirates().isEmpty()) {
            IShip ship = createInitialPiratesShip();
            String firstName = firstNameLoader.getRandomName();
            String lastName = lastNameLoader.getRandomName();
            ISeaPirate pirate = peopleFactory.createNewPirate(firstName, lastName, ship);
            ship.setOwner(pirate);

            piratesState.add(pirate);

        }
    }
    /**
     * Create a ship for the pirate
     * @return initial pirate ship
     */
    private IShip createInitialPiratesShip() {
        int shipChoice = rnd.nextInt(4);
        IShip ship;
        switch (shipChoice) {
            case 0:
                ship = shipFactory.createSnaikka(shipLoader.getRandomName(), EShipUpgrade.LEVEL1, 0); // size of no consequence
                break;
            case 1:
                ship = shipFactory.createCrayer(shipLoader.getRandomName(), EShipUpgrade.LEVEL1, 0);
                break;
            case 2:
                ship = shipFactory.createCog(shipLoader.getRandomName(), EShipUpgrade.LEVEL1, 0);
                break;
            case 3:
                ship = shipFactory.createHolk(shipLoader.getRandomName(), EShipUpgrade.LEVEL1, 0);
                break;
            default:
                throw new RuntimeException("Should never get here");
        }
        ship.setNumberOfSailors(30);
        for (int i = 0; i < 30; i++) {
            shipService.placeWeapon(EWeapon.HAND_WEAPON, ship);
        }
        WeaponSlotCount slotCount = shipService.getWeaponSlotCount(ship.getWeaponSlots());
        int nbBigWeapons = slotCount.getNbLargeSlots()/3;
        int nbSmallWeapons = (slotCount.getNbLargeSlots()*2 + slotCount.getNbSmallSlots())/(2 * 3);
        EWeapon bigWeaponType = getBigWeaponType();
        EWeapon smallWeaponType = getSmallWeaponType();
        for (int i = 0; i < nbBigWeapons; i++) {
            shipService.placeWeapon(bigWeaponType, ship);
        }
        for (int i = 0; i < nbSmallWeapons; i++) {
            shipService.placeWeapon(smallWeaponType, ship);
        }
        return ship;
    }

	/**
	 * Check if the vessel should be handled by this engine. Only ships
	 * with hissed pirate flag that do not belong to a player should be considered.
	 * @param vessel to be checked
	 * @return true if the vessel is flaged as pirate ship and the owner is not a player.
     */
	private boolean handlePirateShip(INavigableVessel vessel) {
		return vessel.getPirateFlag() && !(vessel.getOwner() instanceof IPlayer);
	}

    private EWeapon getBigWeaponType() {
        int year = date.getCurrentDate().getYear();
        if (year <= 1333) {
            return EWeapon.BALLISTA_BIG;
        } else if (year <= 1367) {
            return EWeapon.TREBUCHET_BIG;
        } else {
            return EWeapon.BOMBARD;
        }
    }
    private EWeapon getSmallWeaponType() {
        int year = date.getCurrentDate().getYear();
        if (year <= 1333) {
            return EWeapon.BALLISTA_SMALL;
        } else if (year <= 1367) {
            return EWeapon.TREBUCHET_SMALL;
        } else {
            return EWeapon.CANNON;
        }
    }

}
