package ch.sahits.game.openpatrician.engine.sea.model;

import ch.sahits.game.openpatrician.clientserverinterface.model.WaitingTradeMissionWrapper;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.OrlegShipOutfittedEvent;
import ch.sahits.game.openpatrician.clientserverinterface.service.ConvoyService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.event.data.RepairFinishedEvent;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Special mission that blocks all ships of a player from a convoy for them
 * to be repaired and then reform the convoy.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Oct 13, 2018
 */
@ClassCategory({EClassCategory.PROTOTYPE_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class ConvoyRepairMission extends WaitingTradeMissionWrapper {
    @Getter
    private IShip orleg;
    @Getter
    @ListType(IShip.class)
    private List<IShip> ships;
    @Getter
    private boolean publicConvoy;

    private IConvoy reformedConvoy;
    @ListType(IShip.class)
    private List<IShip> repairedShips = new ArrayList<>();

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @XStreamOmitField
    private ConvoyService convoyService;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;
    /**
     * Constructor registering itself as the mission data on the player for all ships.
     * @param orleg
     * @param convoyShips
     */
    public ConvoyRepairMission(IShip orleg, List<IShip> convoyShips, boolean publicConvoy) {
        super(((IAIPlayer) orleg.getOwner()).getTradeMission(orleg));
        this.orleg = orleg;
        this.ships = convoyShips;
        this.publicConvoy = publicConvoy;
        for (IShip convoyShip : convoyShips) {
            IAIPlayer player = (IAIPlayer) convoyShip.getOwner();
            ITradeMissionData origMissionData =
                    player.getTradeMission(convoyShip);
             player.setTradeMission(convoyShip, new WaitingTradeMissionWrapper(origMissionData));
        }
    }

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @Subscribe
    public void handleShipRepaired(RepairFinishedEvent event) {
        INavigableVessel vessel = event.getShip();
        // Is it a ship we are interessted in?
        if (vessel.equals(orleg) || ships.contains(vessel)) {
            IShip ship = (IShip) vessel;
            ICity city = shipService.findCity(ship).get();
            IAIPlayer player = (IAIPlayer) ship.getOwner();
            // is it the orleg? => form convoy check capability
            if (ship.equals(orleg)) {
                Optional<IShip> optOrleg = convoyService.outfitAsOrleg(ship, city);
                if (optOrleg.isPresent()) {
                    reformedConvoy = convoyService.create(this.orleg, publicConvoy);
                }
            } else {
                // is member ship, join queue or convoy
                if (reformedConvoy != null) {
                    convoyService.join(reformedConvoy, ship);
                } else {
                    repairedShips.add(ship);
                }
                ships.remove(ship);
            }
            // is last ship? => remove special missions, unregister, send convoy on the way.
            if (reformedConvoy != null) {
                checkConvoySendingOn(player);
            }
        }
    }

    private void checkConvoySendingOn(IAIPlayer player) {
        if (ships.isEmpty()) {
            for (IShip ship : reformedConvoy.getShips()) {
                ITradeMissionData tradeMissionData = ((WaitingTradeMissionWrapper) player.getTradeMission(ship)).getTradeMissionData();
                player.setTradeMission(ship, tradeMissionData);
            }
            clientServerEventBus.unregister(this);
            player.getTradeStrategyType(reformedConvoy).getStrategy().initializeTradeCycle(player, reformedConvoy);
        }
    }

    /**
     * When a ship has finally been fitted for orlead duties, create the convoy.
     * @param event
     */
    @Subscribe
    public void handleShipFittedForOrleg(OrlegShipOutfittedEvent event) {
        if (event.getShip().equals(orleg)) {
            reformedConvoy = convoyService.create(this.orleg, publicConvoy);
            checkConvoySendingOn((IAIPlayer) reformedConvoy.getOwner());
        }
    }
}
