package ch.sahits.game.openpatrician.engine.sea.model;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import com.carrotsearch.hppc.ObjectDoubleMap;
import com.carrotsearch.hppc.ObjectDoubleScatterMap;
import com.google.common.base.Preconditions;
import javafx.geometry.Point2D;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

/**
 * The graph represents an undirected graph.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 31, 2015
 */
@ClassCategory({EClassCategory.MODEL, EClassCategory.SINGLETON_BEAN})
public class GraphAStar<T> implements Iterable<T> {


    /*
         * A map from the nodeId to outgoing edge.
         * An outgoing edge is represented as a tuple of NodeData and the edge length
         */
    @MapType(key = Point2D.class, value = Map.class)
    private Map<T, ObjectDoubleScatterMap<NodeData<T>>> graph;
    /*
     * A map of heuristic from a node to each other node in the graph.
     */
    @MapType(key = Point2D.class, value = Map.class)
    private final Map<T, ObjectDoubleMap<T>> heuristicMap;
    /*
     * A map between nodeId and nodedata.
     */
    @MapType(key = Point2D.class, value = NodeData.class)
    private Map<T, NodeData<T>> nodeIdNodeData;

    public GraphAStar(Map<T, ObjectDoubleMap<T>> heuristicMap) {
        Preconditions.checkNotNull(heuristicMap, "The huerisic map should not be null");
        graph = new HashMap<>();
        nodeIdNodeData = new HashMap<>();
        this.heuristicMap = heuristicMap;
    }



    /**
     * Adds a new node to the graph.
     * Internally it creates the nodeData and populates the heuristic map concerning input node into node data.
     *
     * @param nodeId the node to be added
     * @param isTargetNode flag indicatin the the node is a target node. Only target nodes are contained in the heuristic.
     */
    public void addNode(T nodeId, boolean isTargetNode) {
        Preconditions.checkNotNull(nodeId, "The node id cannot be null");
        if (!isTargetNode) {
            if (!heuristicMap.containsKey(nodeId))
                throw new NoSuchElementException("This node (" + nodeId + ") is not a part of hueristic map");
        }

        addNodeInternal(nodeId);

//        graph.put(nodeId, new HashMap<>());
//        nodeIdNodeData.put(nodeId, new NodeData<>(nodeId, heuristicMap.get(nodeId)));
    }
    /**
     * Adds a new node to the graph. This is used internally in the initial creation where all nodes
     * are added but only the target nodes have heuristic.
     * Internally it creates the nodeData and populates the heuristic map concerning input node into node data.
     *
     * @param nodeId the node to be added
     */
    public void addNodeInternal(T nodeId) {
        Preconditions.checkNotNull(nodeId, "The node id cannot be null");

        graph.put(nodeId, new ObjectDoubleScatterMap<>());
        nodeIdNodeData.put(nodeId, new NodeData<>(nodeId, heuristicMap.get(nodeId)));
    }

    /**
     * Adds an edge from source node to destination node.
     * There can only be a single edge from source to node.
     * Adding additional edge would overwrite the value
     *
     * @param nodeIdFirst   the first node to be in the edge
     * @param nodeIdSecond  the second node to be second node in the edge
     * @param length        the length of the edge.
     */
    public void addEdge(T nodeIdFirst, T nodeIdSecond, double length) {
        if (nodeIdFirst == null || nodeIdSecond == null) throw new NullPointerException("The first nor second node can be null.");

        if (!heuristicMap.containsKey(nodeIdFirst) || !heuristicMap.containsKey(nodeIdSecond)) {
            throw new NoSuchElementException("Source and Destination both should be part of the part of hueristic map: " + !heuristicMap.containsKey(nodeIdFirst) + " || " + !heuristicMap.containsKey(nodeIdSecond));
        }
        if (!heuristicMap.containsKey(nodeIdFirst) || !heuristicMap.containsKey(nodeIdSecond)) {
            throw new NoSuchElementException("Source and Destination both should be part of the part of hueristic map: " + !heuristicMap.containsKey(nodeIdFirst) + " || " + !heuristicMap.containsKey(nodeIdSecond));
        }        if (!graph.containsKey(nodeIdFirst) || !graph.containsKey(nodeIdSecond)) {
            throw new NoSuchElementException("Source and Destination both should be part of the part of graph");
        }

        graph.get(nodeIdFirst).put(nodeIdNodeData.get(nodeIdSecond), length);
        graph.get(nodeIdSecond).put(nodeIdNodeData.get(nodeIdFirst), length);
    }
    /**
     * Adds an edge from source node to destination node.
     * This method is used for the initial creation of the graph where
     * only the heuristic for the target node exists.
     * There can only be a single edge from source to node.
     * Adding additional edge would overwrite the value
     *
     * @param nodeIdFirst   the first node to be in the edge
     * @param nodeIdSecond  the second node to be second node in the edge
     * @param length        the length of the edge.
     */
    public void addEdgeInternal(T nodeIdFirst, T nodeIdSecond, double length) {
        if (nodeIdFirst == null || nodeIdSecond == null) throw new NullPointerException("The first nor second node can be null.");

        if (!graph.containsKey(nodeIdFirst) || !graph.containsKey(nodeIdSecond)) {
            throw new NoSuchElementException("Source and Destination both should be part of the part of graph");
        }

        graph.get(nodeIdFirst).put(nodeIdNodeData.get(nodeIdSecond), length);
        graph.get(nodeIdSecond).put(nodeIdNodeData.get(nodeIdFirst), length);
    }
    /**
     * Returns immutable view of the edges
     *
     * @param nodeId    the nodeId whose outgoing edge needs to be returned
     * @return          An immutable view of edges leaving that node
     */
    public ObjectDoubleMap<NodeData<T>> edgesFrom (T nodeId) {
        Preconditions.checkNotNull(nodeId, "The node id cannot be null");
        if (!heuristicMap.containsKey(nodeId)) throw new NoSuchElementException("This node is not a part of hueristic map");
        if (!graph.containsKey(nodeId)) throw new NoSuchElementException("The node should not be null.");

        return graph.get(nodeId).clone();

//        return Collections.unmodifiableMap(graph.get(nodeId));
    }

    /**
     * The nodedata corresponding to the current nodeId.
     *
     * @param nodeId    the nodeId to be returned
     * @return          the nodeData from the
     */
    public NodeData<T> getNodeData (T nodeId) {
        Preconditions.checkNotNull(nodeId, "The node id cannot be null");
        if (!nodeIdNodeData.containsKey(nodeId))  { throw new NoSuchElementException("The nodeId does not exist: "+nodeId); }
        return nodeIdNodeData.get(nodeId);
    }

    /**
     * Check if the nodeid is already present in the data.
     *
     * @param nodeId to be looked up.
     * @return true if the node is contained in the graph.
     */
    public boolean containsNode(T nodeId) {
        return nodeIdNodeData.containsKey(nodeId);
    }

    /**
     * Returns an iterator that can traverse the nodes of the graph
     *
     * @return an Iterator.
     */
    @Override
    public Iterator<T> iterator() {
        return graph.keySet().iterator();
    }
    public int size() {
        return graph.size();
    }

}
