package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.engine.event.task.WeaponConstructionTask;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.WeaponProperties;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.weapon.ArmoryRegistry;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Engine controlling the blacksmiths in the armories.
 *
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Jul 28, 2017
 */

@Component
@Lazy
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class BlacksmithEngine extends AbstractEngine {
    private static final EWare[] wares = {EWare.WOOD, EWare.IRON, EWare.LEATHER, EWare.HEMP};

    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private ArmoryRegistry armories;

    @Autowired
    private Date date;
    @Autowired
    private ServerSideTaskFactory taskFactory;
    @Autowired
    private TimedUpdatableTaskList taskList;

    @Autowired
    private WeaponProperties weaponProperties;

    @PostConstruct
    private void init() {
        timerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        timerEventBus.unregister(this);
    }

    @Override
    public List<AbstractEngine> getChildren() {
        return newArrayList();
    }
    @Subscribe
    public void handleDailyUpdate(ClockTickDayChange event) {
        for (Entry<ICity, IArmory> entry : armories) {
            IArmory armory = entry.getValue();
            ICity city = entry.getKey();
            // Is occupied
            if (date.getCurrentDate().isAfter(armory.occupiedUntil())) {
                // Find weapon to produce (include required ware check) priority, availability, wares, capability
                Optional<EWeapon> weaponToProduce = findWeaponToProduce(armory, city);
                //noinspection OptionalIsPresent
                if (weaponToProduce.isPresent()) {
                    EWeapon weapon = weaponToProduce.get();
                    // buy wares
                    for (EWare ware : wares) {
                        int amount = getConstructionAmount(weapon, ware);
                        city.move(ware, -amount, null);
                    }
                    // determine completion date
                    LocalDateTime finished = date.getCurrentDate().plusDays(getBuildDuration(weapon));
                    armory.occupy(finished);
                    // add timed task for completion + experience update.
                    WeaponConstructionTask task = taskFactory.getWeaponConstructionFinishedTask(armory, weapon);
                    task.setExecutionTime(finished);
                    taskList.add(task);
                }
            }
        }
    }

    /**
     * Find the weapon that should be produced. Consider the following criteria:
     * <ol>
     *     <li>capability of the blacksmith: certain weapons can only be produced with a certain experience</li>
     *     <li>available wares to produce the weapons</li>
     *     <li>priority</li>
     *     <li>availability of the weapon in the arsenal: if two weapons can be produced, the one with lower availability
     *     in storage is produced</li>
     * </ol>
     * @param armory of the blacksmith
     * @param city in which the armory is located
     * @return empty if no weapon can be produced, otherwise the weapon that should be produced.
     */
    @VisibleForTesting
    Optional<EWeapon> findWeaponToProduce(IArmory armory, ICity city) {
       List<EWeapon> producableWeapons = newArrayList();
       producableWeapons.addAll(Arrays.asList(EWeapon.values()));
       filterCapability(producableWeapons, armory.getExperience());
       filterWareAvailability(producableWeapons, city);
       filterAndSortPriority(producableWeapons, armory);
       sortWeaponAvailability(producableWeapons, armory);
       if (producableWeapons.isEmpty()) {
           return Optional.empty();
       } else {
           return Optional.of(producableWeapons.get(0));
       }
    }

    private void sortWeaponAvailability(List<EWeapon> producableWeapons, IArmory armory) {
        WeaponAvailabilityComparator comparator = new WeaponAvailabilityComparator(armory);
        producableWeapons.sort(comparator);

    }

    private void filterAndSortPriority(List<EWeapon> producableWeapons, IArmory armory) {
        List<EWeapon> prioWeapons = newArrayList();
        for (EWeapon weapon : producableWeapons) {
            switch (weapon) {
                case SWORD:
                    if (armory.isSwordPriority()) {
                        prioWeapons.add(weapon);
                    }
                    break;
                case BOW:
                    if (armory.isBowPriority()) {
                        prioWeapons.add(weapon);
                    }
                    break;
                case CROSSBOW:
                    if (armory.isCrossbowPriority()) {
                        prioWeapons.add(weapon);
                    }
                    break;
                case MUSKET:
                    if (armory.isMusketPriority()) {
                        prioWeapons.add(weapon);
                    }
                    break;
            }
        }
        if (!prioWeapons.isEmpty()) {
            producableWeapons.clear();
            producableWeapons.addAll(prioWeapons);
        }
    }

    private void filterWareAvailability(List<EWeapon> producableWeapons, ICity city) {
        for (Iterator<EWeapon> iterator = producableWeapons.iterator(); iterator.hasNext(); ) {
            EWeapon weapon = iterator.next();
            boolean remove = false;
            for (EWare ware : wares) {
                int requiredAmount = getConstructionAmount(weapon, ware);
                if (city.getWare(ware).getAmount() < requiredAmount) {
                    remove = true;
                    break;
                }
            }
            if (remove) {
                iterator.remove();
            }
        }
    }

    private void filterCapability(List<EWeapon> producableWeapons, double experience) {
        producableWeapons.removeIf(weapon -> experience < getMinExperience(weapon));
    }

    private double getMinExperience(EWeapon weapon) {
        switch (weapon) {
            case BALLISTA_BIG:
                return weaponProperties.getBallistaBig().getMinExperiance();
            case BALLISTA_SMALL:
                return weaponProperties.getBallistaSmall().getMinExperiance();
            case BOMBARD:
                return weaponProperties.getBombard().getMinExperiance();
            case BOW:
                return weaponProperties.getBow().getMinExperiance();
            case CANNON:
                return weaponProperties.getCanon().getMinExperiance();
            case CROSSBOW:
                return weaponProperties.getCrossbow().getMinExperiance();
            case HAND_WEAPON:
                return weaponProperties.getCutlass().getMinExperiance();
            case MUSKET:
                return weaponProperties.getMusket().getMinExperiance();
            case SWORD:
                return weaponProperties.getSword().getMinExperiance();
            case TREBUCHET_BIG:
                return weaponProperties.getTrebuchetBig().getMinExperiance();
            case TREBUCHET_SMALL:
                return weaponProperties.getTrebuchetSmall().getMinExperiance();
            default:
                return Double.MAX_VALUE;
        }
    }

    private int getConstructionAmount(EWeapon weapon, EWare ware) {
        switch (weapon) {
            case BALLISTA_BIG:
                return (int) Math.rint(getConstructionAmount(EWeapon.BALLISTA_SMALL, ware) * 1.5);
            case BALLISTA_SMALL:
                switch (ware) {
                    case WOOD:
                        return 4;
                    case LEATHER:
                        return 3;
                    case IRON:
                        return 10;
                    case HEMP:
                        return 1;
                    default:
                        return 0;
                }
            case BOMBARD:
                switch (ware) {
                    case WOOD:
                        return 3;
                    case LEATHER:
                        return 2;
                    case IRON:
                        return 15;
                    case HEMP:
                        return 1;
                    default:
                        return 0;
                }
            case BOW:
                switch (ware) {
                    case WOOD:
                        return 1;
                    case LEATHER:
                        return 1;
                    case IRON:
                        return 1;
                    case HEMP:
                        return 1;
                    default:
                        return 0;
                }
            case CANNON:
                switch (ware) {
                    case WOOD:
                        return 3;
                    case LEATHER:
                        return 2;
                    case IRON:
                        return 20;
                    case HEMP:
                        return 1;
                    default:
                        return 0;
                }
            case CROSSBOW:
                switch (ware) {
                    case WOOD:
                        return 1;
                    case LEATHER:
                        return 1;
                    case IRON:
                        return 2;
                    case HEMP:
                        return 1;
                    default:
                        return 0;
                }
            case HAND_WEAPON:
                switch (ware) {
                    case WOOD:
                        return 1;
                    case LEATHER:
                        return 1;
                    case IRON:
                        return 1;
                    case HEMP:
                        return 0;
                    default:
                        return 0;
                }
            case MUSKET:
                switch (ware) {
                    case WOOD:
                        return 1;
                    case LEATHER:
                        return 1;
                    case IRON:
                        return 5;
                    case HEMP:
                        return 0;
                    default:
                        return 0;
                }
            case SWORD:
                switch (ware) {
                    case WOOD:
                        return 0;
                    case LEATHER:
                        return 1;
                    case IRON:
                        return 2;
                    case HEMP:
                        return 0;
                    default:
                        return 0;
                }
            case TREBUCHET_BIG:
                return (int) Math.rint(getConstructionAmount(EWeapon.TREBUCHET_SMALL, ware) * 1.5);

            case TREBUCHET_SMALL:
                switch (ware) {
                    case WOOD:
                        return 3;
                    case LEATHER:
                        return 5;
                    case IRON:
                        return 8;
                    case HEMP:
                        return 2;
                    default:
                        return 0;
                }
        }
        return 0;

    }

    private static class WeaponAvailabilityComparator implements Comparator<EWeapon> {
        private final IArmory armory;

        public WeaponAvailabilityComparator(IArmory armory) {
            this.armory = armory;
        }

        @Override
        public int compare(EWeapon w1, EWeapon w2) {
            return getAmount(w1) - getAmount(w2);
        }

        private int getAmount(EWeapon weapon) {
            switch (weapon) {
                case SWORD:
                    return armory.swordAmountProperty().get();
                case BALLISTA_BIG:
                    return armory.ballistaBigAmountProperty().get();
                case BALLISTA_SMALL:
                    return armory.ballistaSmallAmountProperty().get();
                case BOMBARD:
                    return armory.bombardAmountProperty().get();
                case BOW:
                    return armory.bowAmountProperty().get();
                case CANNON:
                    return armory.canonAmountProperty().get();
                case CROSSBOW:
                    return armory.crossbowAmountProperty().get();
                case HAND_WEAPON:
                    return armory.cutlassAmountProperty().get();
                case MUSKET:
                    return armory.musketAmountProperty().get();
                case TREBUCHET_BIG:
                    return armory.trebuchetBigAmountProperty().get();
                case TREBUCHET_SMALL:
                    return armory.trebuchetSmallAmountProperty().get();
                default:
                    return 0;
            }
        }
    }

    private int getBuildDuration(EWeapon weapon) {
        switch (weapon) {
            case SWORD:
                return weaponProperties.getSword().getDuration();
            case BALLISTA_BIG:
                return weaponProperties.getBallistaBig().getDuration();
            case BALLISTA_SMALL:
                return weaponProperties.getBallistaSmall().getDuration();
            case BOMBARD:
                return weaponProperties.getBombard().getDuration();
            case  BOW:
                return weaponProperties.getBow().getDuration();
            case CANNON:
                return weaponProperties.getCanon().getDuration();
            case  CROSSBOW:
                return weaponProperties.getCrossbow().getDuration();
            case HAND_WEAPON:
                return weaponProperties.getCutlass().getDuration();
            case MUSKET:
                return weaponProperties.getMusket().getDuration();
            case TREBUCHET_BIG:
                return weaponProperties.getTrebuchetBig().getDuration();
            case TREBUCHET_SMALL:
                return weaponProperties.getTrebuchetSmall().getDuration();
            default:
                return 0;
        }
    }
}
