package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.annotations.VisibleForTesting;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.List;
import java.util.Random;

import static java.util.stream.Collectors.toList;

/**
 * This strategy builds the biggest possible ship type in a city with a trading office.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jul 30, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class TradingOfficeRandomShipConstructionSelectionStrategy extends RandomShipConstructionSelectionStrategy {
    @Autowired
    private IMap map;
    @Autowired
    private Random rnd;
    @Autowired
    private ApplicationContext context;

    public TradingOfficeRandomShipConstructionSelectionStrategy() {
        super(100000, 50, 15000);
    }

    /**
     * {@inheritDoc}
     * The build location is always the players hometown.
     * @param player
     * @return
     */
    @Override
    public ICity getBuildLocation(IAIPlayer player) {
        return player.getHometown();
    }

    @Override
    public void initShipConstruction(IAIPlayer player, INavigableVessel vessel, EShipType shipType) {
        CollectConstructionWareStrategy strategy = null;
        ICity constructionCity = calculateCostructionCity(player);
        switch (shipType) {
            case HOLK:
                strategy = context.getBean(CollectConstructionTradingOfficeHolkWareStrategy.class);
                break;
            case COG:
                strategy = context.getBean(CollectConstructionTradingOfficeCogWareStrategy.class);
                break;
            case CRAYER:
                strategy = context.getBean(CollectConstructionTradingOfficeCrayerWareStrategy.class);
                break;
            case SNAIKKA:
                strategy = context.getBean(CollectConstructionTradingOfficeSnaikkaWareStrategy.class);
                break;
        }
        ((IAICityDependentShipConstructionStrategy)strategy).setConstructionCity(constructionCity);
        strategy.initializeTradeCycle(player, vessel);
    }

    @VisibleForTesting
    ICity calculateCostructionCity(IAIPlayer player) {
        List<ICity> chosableCities = map.getCities().stream()
                .filter(city -> player.findTradingOffice(city).isPresent())
                .collect(toList());
        return chosableCities.get(rnd.nextInt(chosableCities.size()));
    }
}
