package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Trade step aggregating several wares to buy.
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class AggregatedBuyTradeStep implements ITradeStep {
    @Setter
    private INavigableVessel vessel;
    @Setter
    private ICity city;
    @Setter
    private boolean executeNext = true;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;
    @ListType(BuyTradeStep.class)
    private ArrayList<BuyTradeStep> tradeSteps = new ArrayList<>();
    @Setter
    private long retainCash = 0L;

    public void addBuyStep(IWare ware, int maxPrice) {
        BuyTradeStep buyStep = context.getBean(BuyTradeStep.class);
        buyStep.setExecuteNext(executeNext);
        buyStep.setMaxBuyPrice(maxPrice);
        buyStep.setCity(city);
        buyStep.setVessel(vessel);
        buyStep.setWare(ware);
        buyStep.setRetainCash(retainCash);
        tradeSteps.add(buyStep);
    }

    /**
     * Sort the list of wares that should be bought so that the
     * wares with the least amount are at the begining of the list.
     * @return priority sorted ware list.
     */
    private List<BuyTradeStep> getPriorityList() {
        ArrayList<BuyTradeStep> list = new ArrayList<>();
        Map<BuyTradeStep,Integer> loadedWares = new HashMap<>();
        for (BuyTradeStep step : tradeSteps) {
            IWare ware = step.getWare();
            int amount = vessel.getWare(ware).getAmount();
            loadedWares.put(step, amount);
        }
        loadedWares.entrySet()
                .stream()
                .sorted(Map.Entry.<BuyTradeStep, Integer>comparingByValue())
                .forEach(entry -> list.add(entry.getKey()));
        return list;
    }

    @Override
    public boolean execute() {
        Preconditions.checkArgument(vessel.getLocation().equals(city.getCoordinates()), "The vessel is not in city "+city.getName()+" but at "+vessel.getLocation());
        List<BuyTradeStep> priorityList = getPriorityList();
        long accumulatedCosts = 0;
        IPlayer player = (IPlayer) vessel.getOwner();
        long initialCash = player.getCompany().getCash();
        for (BuyTradeStep buyTradeStep : priorityList) {
            buyTradeStep.execute();
            accumulatedCosts += buyTradeStep.getCost();
            if (vessel.getCapacity() == 0 || initialCash + accumulatedCosts < 5000) {
                break;
            }
        }
        return executeNext;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        tradeSteps.stream().forEach(step -> sb.append(step.getWare().name()).append(" "));
        return "AggregatedBuyTradeStep"+" player "+ ((IPlayer)vessel.getOwner()).getUuid() +" in "+city.getName()+" buying: "+ sb.toString();
    }
}
