package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.player.IAIShipUpgradeStrategy;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Check if the ship needs to be upgraded. If the vessel is a convoy there will
 * not be an upgrade.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Nov 04, 2017
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class UpgradeShipTradeStep implements ITradeStep {
    @Setter
    private INavigableVessel vessel;
    @Setter
    private IAIPlayer player;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;

    /**
     * {@inheritDoc}
     * @return false if an upgrade is executed. The wake up event is {@link ch.sahits.game.openpatrician.event.data.RefitFinishedEvent}.
     */
    @Override
    public boolean execute() {
        if (vessel instanceof IShip) {
            IShip ship = (IShip) vessel;
            IAIShipUpgradeStrategy strategy = player.getShipUpgradeStrategyType().getStrategy();
            if (strategy.shouldUpgrade(ship)) {
                ICity city = shipService.findCity(ship).get();
                strategy.upgrade(ship, city);
                return false;
            }
        }
        return true;
    }
    @Override
    public String toString() {
        ICity city = shipService.findCity(vessel).get();
        return "UpgradeShipTradeStep "+ ((IPlayer)vessel.getOwner()).getUuid() +" in "
                +city.getName()+" "
                +((IAIPlayer)vessel.getOwner()).getShipUpgradeStrategyType();
    }
}
