package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.openpatrician.event.data.ShipPositionUpdateEvent;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.EMessageCategory;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.cityhall.IAcceptedAldermanTask;
import ch.sahits.game.openpatrician.model.city.cityhall.ICapturePirateNest;
import ch.sahits.game.openpatrician.model.city.cityhall.impl.AldermanOffice;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.map.MapProperties;
import ch.sahits.game.openpatrician.model.map.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.map.PirateMapSegmentImage;
import ch.sahits.game.openpatrician.model.map.TreasureMapSegmentImage;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.server.MapLocationDetectionModel;
import ch.sahits.game.openpatrician.model.server.MapSegmentDataCheck;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 07, 2015
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
public class MapLocationDetector {
    @Autowired
    private MapLocationDetectionModel dataModel;

    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("syncServerClientEventBus")
    private EventBus syncServerClientEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private Date date;
    @Autowired
    private AldermanOffice aldermanOffice;
    @Autowired
    private IMap map;
    @Autowired
    private Random rnd;
    @Autowired
    private LocationTracker locationTracker;
    @Autowired
    private MapProperties mapProperties;

    private boolean shipsUpdated = false;

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
        timerEventBus.register(this);
        syncServerClientEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
        timerEventBus.unregister(this);
        syncServerClientEventBus.unregister(this);
    }


    @Subscribe
    public void checkShipUpdate(ShipPositionUpdateEvent event) {
        shipsUpdated = true;
    }
    @VisibleForTesting
    void handleShipLocationUpdate() {
        for (MapSegmentedImage mapSegment : dataModel.getMapSegments()) {
            Point2D location = mapSegment.getLocation();
            List<INavigableVessel> shipsToCheck = locationTracker.getShipsInSegments(location, mapProperties.getVisibilityDistance());
            for (INavigableVessel ship : shipsToCheck) {
                if (location.distance(ship.getLocation()) <= mapProperties.getVisibilityDistance() && !dataModel.isInCheckedList(ship)) {
                    LocalDateTime dealine = date.getCurrentDate().plusDays(1);
                    MapSegmentDataCheck check = new MapSegmentDataCheck(ship, mapSegment, dealine);
                    dataModel.getCheckList().put(check.getShip(), check);
                }
            }
        }
    }

    @Subscribe
    public void checkShipLeaving(ClockTick tick) {
        if (shipsUpdated) {
            handleShipLocationUpdate();
            shipsUpdated = false;
        }
        for (Iterator<INavigableVessel> iterator = dataModel.getCheckList().keySet().iterator(); iterator.hasNext(); ) {
            INavigableVessel ship = iterator.next();
            MapSegmentDataCheck check = dataModel.getCheckList().get(ship);
            if (ship.getLocation().distance(check.getMapSegment().getLocation()) > mapProperties.getVisibilityDistance()) {
                iterator.remove();
            } else { // still in the radius
                LocalDateTime now = date.getCurrentDate();
                if (now.isAfter(check.getDeadline())) {
                    iterator.remove();
                    IPlayer owner = (IPlayer) ship.getOwner();
                    if (check.getMapSegment() instanceof TreasureMapSegmentImage) {
                        TreasureMapSegmentImage treasureMap = (TreasureMapSegmentImage) check.getMapSegment();

                        // post a message
                        if (owner instanceof IHumanPlayer) {
                            DisplayMessage msg = new DisplayMessage(EMessageCategory.TRADE, "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.treasure", new Object[]{treasureMap.getTreasureAmount()});
                            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
                            clientServerEventBus.post(displayMessage);
                            owner.getCompany().updateCash(treasureMap.getTreasureAmount());
                        } else {
                            owner.getCompany().updateCashDirectly(treasureMap.getTreasureAmount());
                        }
                        owner.setSegmentedMap(null);
                        dataModel.removeSegment(treasureMap);
                    }
                    if (check.getMapSegment() instanceof PirateMapSegmentImage) {
                        PirateMapSegmentImage pirateNest = (PirateMapSegmentImage) check.getMapSegment();
                        // calculate if pirate can be overpowered
                        int survivingShips;
                        int pirateFortification = 1;
                        if (ship instanceof IConvoy) {
                            IConvoy convoy = (IConvoy) ship;
                            int randomPirateDestroy = rnd.nextInt(1);
                            survivingShips = convoy.getShips().size() - (pirateNest.getAmountOfShips() + pirateFortification - randomPirateDestroy);

                        } else {
                            int randomPirateDestroy = rnd.nextInt(1);
                            int nbShips = 1;
                            survivingShips =  nbShips - (pirateNest.getAmountOfShips() + pirateFortification - randomPirateDestroy);
                        }
                        if (survivingShips > 0) {
                            dataModel.removeSegment(pirateNest);
                            DisplayMessage msg = new DisplayMessage(EMessageCategory.TRADE, "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.pirateDefeated", new Object[]{pirateNest.getName()});
                            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
                            clientServerEventBus.post(displayMessage);
                            if (ship instanceof IConvoy) {
                                removeDefeatedShips((IConvoy) ship, survivingShips);
                            }
                            List<IAcceptedAldermanTask> acceptedTasks = aldermanOffice.getWorkedOnTasks();
                            for (Iterator<IAcceptedAldermanTask> taskIterator = acceptedTasks.iterator(); taskIterator.hasNext(); ) {
                                IAcceptedAldermanTask task = taskIterator.next();
                                if (task.getPlayer().equals(ship.getOwner())) {
                                    if (task.getTask() instanceof ICapturePirateNest && ((ICapturePirateNest) task.getTask()).getPirateNestMap().equals(pirateNest)) {
                                        taskIterator.remove();
                                        if (task.getDeadline().isAfter(now)) {
                                            for (ICity city : map.getCities()) {
                                                IReputation reputation = city.getReputation(owner);
                                                reputation.update(150);
                                            }
                                        }
                                    }
                                }
                            }
                        } else { // all ships lost
                            DisplayMessage msg = new DisplayMessage(EMessageCategory.TRADE, "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.defeatedByPirate", new Object[]{pirateNest.getName()});
                            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
                            clientServerEventBus.post(displayMessage);
                            if (ship instanceof IConvoy) {
                                final IConvoy convoy = (IConvoy) ship;
                                removeDefeatedShips(convoy, 0);
                            } else {
                                owner.getFleet().remove(ship);
                            }
                        }
                    } // end pirate map

                }
            } // end still in radius
        }
    }
    @VisibleForTesting
    void removeDefeatedShips(IConvoy ship, int survivingShips) {
        List<IShip> ships = ship.getShips();
        int destroyedShips = ships.size() - survivingShips;
            if (destroyedShips < ships.size()) {
                ships.remove(ship.getOrlegShip());
                Collections.shuffle(ships);
            } else {
                ((IPlayer) ship.getOwner()).getFleet().remove(ship);
            }
            Multimap<IPlayer, IShip> destoyedShipsMap = ArrayListMultimap.create();
            while (destroyedShips > 0) {
                IShip s = ships.get(destroyedShips-1);
                destoyedShipsMap.put((IPlayer) s.getOwner(), s);
                ship.removeShip(s);
                ((IPlayer)s.getOwner()).getFleet().remove(s);
                destroyedShips--;
            }
            for (IPlayer player : destoyedShipsMap.keySet()) {
                if (player instanceof IHumanPlayer) {
                    ArrayList<IShip> destroyedShipsOfPlayer = new ArrayList<>(destoyedShipsMap.get(player));
                    if (destroyedShipsOfPlayer.size() > 1) {
                        String titleKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.manytitle";
                        String messageKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.manySunk";
                        StringBuilder sunkenShips = new StringBuilder();
                        for (int i = 0; i < destroyedShipsOfPlayer.size() - 1; i++) {
                            IShip s = destroyedShipsOfPlayer.get(i);
                            sunkenShips.append(s.getName());
                            if (i < destroyedShipsOfPlayer.size() - 2) {
                                sunkenShips.append(", ");
                            }
                        }
                        String lastShipName =  destroyedShipsOfPlayer.get(destroyedShipsOfPlayer.size() - 1).getName();
                        Object[] args = new Object[]{sunkenShips.toString(), lastShipName};
                        DialogTemplate template = DialogTemplate.builder()
                                .closable(true)
                                .titleKey(titleKey)
                                .messageKey(messageKey)
                                .messageArgs(args)
                                .build();
                        DisplayTemplateMessage message = new DisplayTemplateMessage(EMessageCategory.TRADE, titleKey, template);
                        TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)player, message);
                        clientServerEventBus.post(tagetDisplayMsg);
                    } else {
                        String titleKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.title";
                        String messageKey = "ch.sahits.game.openpatrician.engine.sea.MapLocationDetector.shiploss.oneSunk";
                        String lastShipName =  destroyedShipsOfPlayer.get(0).getName();
                        Object[] args = new Object[]{lastShipName};
                        DialogTemplate template = DialogTemplate.builder()
                                .closable(true)
                                .titleKey(titleKey)
                                .messageKey(messageKey)
                                .messageArgs(args)
                                .build();
                        DisplayTemplateMessage message = new DisplayTemplateMessage(EMessageCategory.TRADE, titleKey, template);
                        TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)player, message);
                        clientServerEventBus.post(tagetDisplayMsg);
                    }
                }
            }
    }


}
