package ch.sahits.game.javafx.control;

import javafx.beans.binding.DoubleBinding;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.scene.image.ImageView;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;

/**
 * If the image within an image view has transparent portions, the click event
 * is only fired when the click happened on a non trasparent part. This class
 * adds a transparent layer for the image view considering the layout and scale
 * dimensions of the image view.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 03, 2015
 * @link https://community.oracle.com/thread/3644092
 */
public class TransparentImageViewClickableArea extends Rectangle{
    private final ImageView imageView;
    private DoubleBinding layoutXBinding = null;
    private DoubleBinding layoutYBinding = null;

    public TransparentImageViewClickableArea(ImageView imageView) {
        super(imageView.getLayoutX(), imageView.getLayoutY(), imageView.getImage().getWidth(), imageView.getImage().getHeight());
        setOnMouseReleased(imageView.getOnMouseReleased());
        this.imageView = imageView;
        setFill(Color.TRANSPARENT);
        bindBounds();
    }

    private void bindBounds() {
        imageView.layoutXProperty().addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue, Number number, Number t1) {
                if (layoutXBinding == null &&
                        imageView.layoutXProperty().isBound() &&
                        imageView.scaleXProperty().isBound()) {
                    initLayoutXBinding();
                }
            }
        });
        imageView.scaleXProperty().addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue, Number number, Number t1) {
                if (layoutXBinding == null &&
                        imageView.layoutXProperty().isBound() &&
                        imageView.scaleXProperty().isBound()) {
                    initLayoutXBinding();
                }
            }
        });
        imageView.layoutYProperty().addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue, Number number, Number t1) {
                if (layoutYBinding == null &&
                        imageView.layoutYProperty().isBound() &&
                        imageView.scaleYProperty().isBound()) {
                    initLayoutYBinding();
                }
            }
        });
        imageView.scaleYProperty().addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue, Number number, Number t1) {
                if (layoutYBinding == null &&
                        imageView.layoutYProperty().isBound() &&
                        imageView.scaleYProperty().isBound()) {
                    initLayoutYBinding();
                }
            }
        });

        widthProperty().bind(imageView.getImage().widthProperty().multiply(imageView.scaleXProperty()));
        heightProperty().bind(imageView.getImage().heightProperty().multiply(imageView.scaleYProperty()));
    }

    private void initLayoutYBinding() {
        layoutYBinding = new DoubleBinding() {
            {
                super.bind(imageView.layoutYProperty(), imageView.scaleYProperty());
            }
            @Override
            protected double computeValue() {
                final double imgHeight = imageView.getImage().getHeight();
                final double actualHeight = imgHeight * imageView.getScaleY();
                double diff = imgHeight - actualHeight;
                return imageView.getLayoutY() + diff/2;
            }
        };
        layoutYProperty().bind(layoutYBinding);
    }

    private void initLayoutXBinding() {
        layoutXBinding = new DoubleBinding() {
            {
                super.bind(imageView.layoutXProperty(), imageView.scaleXProperty());
            }
            @Override
            protected double computeValue() {
                final double imgWidth = imageView.getImage().getWidth();
                final double actualWidth = imgWidth * imageView.getScaleX();
                double diff = imgWidth - actualWidth;
                return imageView.getLayoutX() + diff/2;
            }
        };
        layoutXProperty().bind(layoutXBinding);
    }
}
