package ch.sahits.game.javafx.service;

import ch.sahits.game.graphic.image.impl.XMLImageLoader;
import ch.sahits.game.javafx.control.BaleAmount;
import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.CoinPrice;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.util.StringUtils;
import com.google.common.base.Preconditions;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.StackPane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;
import javafx.scene.shape.Shape;
import javafx.scene.text.Font;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.List;
import java.util.Map;

/**
 * Factory method to create a decorated text from a template.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Nov 10, 2014
 */
public class DecoratedTextFactory {

    public static final String COIN_ICON = "icon:Coin";
    public static final String BARREL_ICON = "icon:Barrel";
    public static final String BALE_ICON = "icon:Bale";
    public static final String SEGMENT_IMAGE = "image:Segment";
    public static final String IMAGE = "image";
    private final Font font;
    private final int leftInset;
    private final int topInset;
    private final int wrappingWidth;
    @Autowired
    @Qualifier("mainScreenXMLImageLoader")
    private XMLImageLoader imageLoader;

    public DecoratedTextFactory(Font font, int leftInset, int topInset, int wrappingWidth) {
        this.font = font;
        this.leftInset = leftInset;
        this.topInset = topInset;
        this.wrappingWidth = wrappingWidth;
    }

    /**
     * Create a decorated text from the template string. The template string can contain placeholders,
     * that are replaced through objects in the parameters Map. The normal replacement uses the toString()
     * method on the map entry's value. There are some special cases where a specific Node is created to
     * represent the placeholder.<br/>
     * Special placeholders:
     * <ul>
     *     <li>icon:Coin: Coin icon followed by the amount passed in as parameter ([icon:Coin coinAmount])</li>
     *     <li>icon:Barrel: Barrel icon followed by the amount passed in as a parameter ([icon:Barrel amount])</li>
     *     <li>icon:Bale: Bale icon followed by the amount passed in as a parameter ([icon:Bale amount])</li>
     *     <li>image:Segment: segmented image split up in 9 segments and only the specified parts are visible ([image:Segment mapSegment1.png visibleSegments])</li>
     *     <li>image: centered image ([image imageName])</li>
     * </ul>
     * The amounts for Coin Barrel and Bale may also be already provided, e.g replaced as part of the message retrevial.
     * In this case it is important that the amount is an unformatted integer. ([icon:Coin{0, number, #}]).
     * @param template template string
     * @param parameters Map containing the parameters.
     * @return
     */
    public DecoratedText createDecoratedText(String template, Map<String, Object> parameters) {
        DecoratedText text = new DecoratedText(font);
        text.setLayoutY(topInset);
        text.setLayoutX(leftInset);
        text.setWrappingWidth(wrappingWidth);

        String[] lines = template.split("\n");
        for (int i = 0; i < lines.length; i++) {
            String curTemplate = lines[i];
            if (i == lines.length -1 && template.endsWith("\n") || i < lines.length -1){
                curTemplate += "\n";
            }
            text = replaceParameters(text, curTemplate, parameters);
        }


        return text;
    }

    /**
     * Replace the placeholders in the template with the values from the parameters object
     * @param template string containing placeholders
     * @param parameters to replace the parameters
     * @return finalized string
     */
    public String replaceTextParameters(String template, Map<String, Object> parameters) {
        int start = template.indexOf('[');
        int end = template.indexOf(']');
        if (start >= 0 && end > start) {
            String head = template.substring(0, start);
            String tail = template.substring(end + 1);
            if (tail.startsWith(" ")) {
                tail = tail.substring(1);
            }
            String key = template.substring(start + 1, end);
            Preconditions.checkNotNull(parameters, "Parameters object may not be null");

            Preconditions.checkArgument(parameters.containsKey(key), "Parameters object does not contain key: "+key);
            return replaceSimpleParameter(parameters, head, tail, key);
        }
        return template;
    }

    private DecoratedText replaceParameters(DecoratedText text, String template, Map<String, Object> parameters) {
        int start = template.indexOf('[');
        int end = template.indexOf(']');
        if (start >= 0 && end > start) {
            String head = template.substring(0, start);
            String tail = template.substring(end + 1);
            if (tail.startsWith(" ")) {
                tail = tail.substring(1);
            }
            String key = template.substring(start + 1, end);
            Preconditions.checkNotNull(parameters, "Parameters object may not be null");
            if (isSpecialKey(key)) {
                text.append(head);
                return replaceSpecialParameter(text, parameters, tail, key);
            } else {
                Preconditions.checkArgument(parameters.containsKey(key), "Parameters object does not contain key: "+key);
                return replaceSimpleParameter(text, parameters, head, tail, key);
            }
        }
        text.append(template);
        return text;
    }

    private DecoratedText replaceSpecialParameter(DecoratedText text, Map<String, Object> parameters, String tail, String key) {
          String keyType = key.substring(0, key.indexOf(" "));
        Node specialNode;
        switch (keyType) {
            case COIN_ICON: {
                specialNode = new CoinPrice(font);
                String paramKey = key.substring(key.indexOf(" ") + 1);
                if (StringUtils.isInteger(paramKey)) {
                    ((CoinPrice) specialNode).setAmount(Integer.parseInt(paramKey));
                } else {
                    Preconditions.checkArgument(!parameters.containsKey(key), "Parameters object does not contain key: " + key);
                    ((CoinPrice) specialNode).setAmount((int) parameters.get(paramKey));
                }
                break;
            }
            case BARREL_ICON: {
                specialNode = new BarrelAmount(font);
                String paramKey = key.substring(key.indexOf(" ") + 1);
                if (StringUtils.isInteger(paramKey)) {
                    ((BarrelAmount) specialNode).setAmount(Integer.parseInt(paramKey));
                } else {
                    Preconditions.checkArgument(!parameters.containsKey(key), "Parameters object does not contain key: " + key);
                    ((BarrelAmount) specialNode).setAmount((int) parameters.get(paramKey));
                }
                break;
            }
            case BALE_ICON: {
                specialNode = new BaleAmount(font);

                String paramKey = key.substring(key.indexOf(" ") + 1);
                if (StringUtils.isInteger(paramKey)) {
                    ((BaleAmount) specialNode).setAmount(Integer.parseInt(paramKey));
                } else {
                    Preconditions.checkArgument(!parameters.containsKey(key), "Parameters object does not contain key: " + key);
                    ((BaleAmount) specialNode).setAmount((int) parameters.get(paramKey));
                }
                break;
            }
            case SEGMENT_IMAGE: {
                String[] params = key.split(" ");
                String imageName = params[1];
                String paramName = params[2];
                Preconditions.checkArgument(!parameters.containsKey(key), "Parameters object does not contain key: "+key);
                specialNode = new StackPane();
                Rectangle ph = new Rectangle(wrappingWidth, 1, Color.TRANSPARENT);
                Image img = imageLoader.getImage(imageName);
                ImageView imgView = new ImageView(img);
                Shape clip = createSegmentedShape(img.getWidth(), img.getHeight(), (List<Integer>) parameters.get(paramName));
                imgView.setClip(clip);
                ((StackPane)specialNode).getChildren().addAll(ph, imgView);
                break;
            }
            case IMAGE: {
                String[] params = key.split(" ");
                String imageName = params[1];
                specialNode = new StackPane();
                Rectangle ph = new Rectangle(wrappingWidth, 1, Color.TRANSPARENT);
                Image img = imageLoader.getImage(imageName);
                ImageView imgView = new ImageView(img);
                ((StackPane)specialNode).getChildren().addAll(ph, imgView);
                break;
            }
            default:
                throw new IllegalArgumentException("Unknown key" + keyType);
        }
        text.append(specialNode);
        return replaceParameters(text, tail, parameters);
    }

    private Shape createSegmentedShape(double width, double heigth, List<Integer> visibleSegments) {
        double segmentWidth = width/3;
        double segmentHeight = heigth/3;
        Shape shape = new Rectangle();
        for (Integer visibleSegment : visibleSegments) {
            Rectangle rect = new Rectangle();
            rect.setWidth(segmentWidth);
            rect.setHeight(segmentHeight);
            int row = (visibleSegment-1) / 3;
            int col = (visibleSegment-1) % 3;
            rect.setY(row * segmentHeight);
            rect.setX(col * segmentWidth);
            shape = Shape.union(shape, rect);
        }
        return shape;
    }

    private boolean isSpecialKey(String key) {
        if (key.startsWith(COIN_ICON)) {
            return true;
        }
        if (key.startsWith(BARREL_ICON)) {
            return true;
        }
        if (key.startsWith(BALE_ICON)) {
            return true;
        }
        if (key.startsWith(SEGMENT_IMAGE)) {
            return true;
        }
        if (key.startsWith(IMAGE)) {
            return true;
        }
        return false;
    }

    private DecoratedText replaceSimpleParameter(DecoratedText text, Map<String, Object> parameters, String head, String tail, String key) {
        if (tail.startsWith("\n")) {
            text.append(head + parameters.get(key) + "\n");
            tail = tail.substring(1);
            if (tail.length() > 0) {
                return replaceParameters(text, tail, parameters);
            } else {
                return text;
            }
        } else {
            text.append(head + parameters.get(key));
            if (tail.length() > 0) {
                return replaceParameters(text, tail, parameters);
            } else {
                return text;
            }
        }
    }
    private String replaceSimpleParameter(Map<String, Object> parameters, String head, String tail, String key) {
        StringBuffer text = new StringBuffer();
        if (tail.startsWith("\n")) {
            text.append(head + parameters.get(key) + "\n");
            tail = tail.substring(1);
            if (tail.length() > 0) {
                text.append(tail);
                return replaceTextParameters(text.toString(), parameters);
            } else {
                return text.toString();
            }
        } else {
            text.append(head + parameters.get(key) + " ");
            if (tail.length() > 0) {
                text.append(tail);
                return replaceTextParameters(text.toString(), parameters);
            } else {
                return text.toString();
            }
        }
    }


}
