package ch.sahits.game.javafx.control;

import java.net.URL;

import javafx.beans.property.DoubleProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.ReadOnlyDoubleProperty;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.beans.property.ReadOnlyStringProperty;
import javafx.beans.property.StringProperty;
import javafx.event.EventHandler;
import javafx.geometry.Insets;
import javafx.scene.Node;
import javafx.scene.control.Control;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Font;
import ch.sahits.game.openpatrician.util.model.Text;
/**
 * Pagination control that allows for input text to be spit up on the pages based
 * on the amount of text that can be displayed on one page.
 * The size of the page is controlled by the properties <code>contentMaxWidth</code>
 * and <code>contentMaxHeight</code>.
 * Furthermore this control allows you to define a node for the back and next action.
 * This is especially helpful in the case when applying a transformation on this control.
 * If the properties <code>backButtonNode</code> and <code>nextButtonNode</code>
 * are not set, the next and back action will be mapped on the labels.
 * The contents of this Pagination does not react on resizing the component. Instead the size 
 * of the contents is controlled by the properties {@link #contentMaxHeightProperty()} and
 * {@link #contentMaxWidthProperty()}. However the control will resize itself by expanding to the
 * right and the bottom. To avoid this pack the Pagination into a group and set its managable flag
 * to false.
 * <br/>
 * To include this control in the workflow together with other controls you can define an event handler
 * which will then be executeted when you hit the back button on the first page or the next button on the
 * last page.<br/>
 * This control can only be instantiated using it's builder.
 *  
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Aug 4, 2013
 *
 */
public class PaginationV2 extends Control {
    private final StringProperty backButtonLabel;
    private final StringProperty nextButtonLabel;
    // Optional
	private final ObjectProperty<Node> backButtonNode;
	private final ObjectProperty<Node> nextButtonNode;
	private final ObjectProperty<EventHandler<MouseEvent>> firstLastPageAction;

	private final ObjectProperty<Text> text;

	private final DoubleProperty contentMaxWidth;
	private final DoubleProperty contentMaxHeight;
	private final ObjectProperty<Insets> paddingLocal;

    // TODO aho Aug 17, 2013: With Java 8 define a font property on the Pagination itself
	private final ReadOnlyObjectProperty<Font> navigationLabelFont;
	private final ReadOnlyObjectProperty<Font> headerFont;
	private final ReadOnlyObjectProperty<Font> paragraphFont;
	private final ReadOnlyObjectProperty<Font> centeredFont;

	public PaginationV2(StringProperty backButtonLabel,
			StringProperty nextButtonLabel,
			ObjectProperty<Node> backButtonNode,
			ObjectProperty<Node> nextButtonNode,
			ObjectProperty<EventHandler<MouseEvent>> firstLastPageAction,
			ObjectProperty<Text> text, DoubleProperty contentMaxWidth,
			DoubleProperty contentMaxHeight, ObjectProperty<Insets> paddingLocal,
			ReadOnlyObjectProperty<Font> navigationLabelFont,
			ReadOnlyObjectProperty<Font> headerFont,
			ReadOnlyObjectProperty<Font> paragraphFont,
			ReadOnlyObjectProperty<Font> centeredFont) {
		super();
		this.backButtonLabel = backButtonLabel;
		this.nextButtonLabel = nextButtonLabel;
		this.backButtonNode = backButtonNode;
		this.nextButtonNode = nextButtonNode;
		this.firstLastPageAction = firstLastPageAction;
		this.text = text;
		this.contentMaxWidth = contentMaxWidth;
		this.contentMaxHeight = contentMaxHeight;
		this.paddingLocal = paddingLocal;
		this.navigationLabelFont = navigationLabelFont;
		this.headerFont = headerFont;
		this.paragraphFont = paragraphFont;
		this.centeredFont = centeredFont;

        this.getStyleClass().add(this.getClass().getSimpleName());

	}

	@Override
    protected String getUserAgentStylesheet() {
        URL cssURL = getClass().getResource("/ch/sahits/game/javafx/control/"+getClass().getSimpleName()+".css");
		return cssURL.toExternalForm();
    }
    
    public double getContentMaxHeight() {
        return contentMaxHeight.get();
    }

    /**
     * Retrieve the maximal height of the content
     * @return 
     */
    public ReadOnlyDoubleProperty contentMaxHeightProperty() {
        return contentMaxHeight;
    }

    public double getContentMaxWidth() {
        return contentMaxWidth.get();
    }

    /**
     * Property containing the maximal width of the content.
     * @return 
     */
    public ReadOnlyDoubleProperty contentMaxWidthProperty() {
        return contentMaxWidth;
    }

    public Text getText() {
        return text.get();
    }

    /**
     * Property containing the text content of this
     * control.
     * @return 
     */
    public ReadOnlyObjectProperty<Text> textProperty() {
        return text;
    }

    public String getNextButtonLabel() {
        return nextButtonLabel.get();
    }

    /**
     * Property defining the label of the next button.
     * Default is 'next'
     * @return 
     */
    public ReadOnlyStringProperty nextButtonLabelProperty() {
        return nextButtonLabel;
    }

    public String getBackButtonLabel() {
        return backButtonLabel.get();
    }

    /**
     * Property defining the label of the back button.
     * Default is 'back'
     * @return 
     */
    public ReadOnlyStringProperty backButtonLabelProperty() {
        return backButtonLabel;
    }
    
    
    /**
     * Retrieve the node definition to be used for the back button.
     * This method is used by the Skin.
     * @return 
     */
    public final Node getBackButtonNode() {
        return backButtonNode.get();
    }
    
    public Insets getPaddingLocal() {
    	return paddingLocal.get();
    }
    
    public ReadOnlyObjectProperty<Insets> paddingLocalProperty() {
		return paddingLocal;
	}

	/**
     * Retrieve the node definition to be used for the next button.
     * This method is used by the Skin.
     * @return 
     */
    public final Node getNextButtonNode() {
        return nextButtonNode.get();
    }

	public ReadOnlyObjectProperty<EventHandler<MouseEvent>> firstLastPageActionProperty() {
		return firstLastPageAction;
	}
	
	
	public EventHandler<MouseEvent> getFirstLastPageAction() {
		return firstLastPageAction.get();
	}
	
	public ReadOnlyObjectProperty<Font> navigationLabelFontProperty() {
		return navigationLabelFont;
	}
	
	public Font getNavigationLabelFont() {
		return navigationLabelFont.get();
	}

	public ReadOnlyObjectProperty<Font> headerFontProperty() {
		return headerFont;
	}
	
	public Font getHeaderFont() {
		return headerFont.get();
	}
	
	public ReadOnlyObjectProperty<Font> centeredFontProperty() {
		return centeredFont;
	}
	
	public Font getCenteredFont() {
		return centeredFont.get();
	}
	
	public ReadOnlyObjectProperty<Font> paragraphFontProperty() {
		return paragraphFont;
	}
	
	public Font getParagraphFont() {
		return paragraphFont.get();
	}
}
