package ch.sahits.game.javafx.control.skin;

import ch.sahits.game.javafx.control.OpenPatricianSlider;
import ch.sahits.game.javafx.control.TextSizingUtility;
import com.sun.javafx.scene.control.skin.SkinBase;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.event.EventHandler;
import javafx.geometry.Bounds;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.paint.Color;
import javafx.scene.shape.Circle;
import javafx.scene.shape.Line;
import javafx.scene.shape.Rectangle;
import javafx.scene.shape.Shape;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
/**
 * Skin for OpenPatricianSlider.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Sep 24, 2013
 *
 */
public class OpenPatricianSliderSkin extends
		SkinBase<OpenPatricianSlider, OpenPatricianSliderBehavior> {
	// for testability reasons package private
	DoubleProperty currentRotation;
	private TextSizingUtility sizeing;
	private List<Double> angles;
    private final int yTranslation = 15; // might be due to the labels
    
    private final OpenPatricianSlider control;
	
	public OpenPatricianSliderSkin(OpenPatricianSlider slider) {
		super(slider, new OpenPatricianSliderBehavior(slider));
		control = slider;
		
		currentRotation = new SimpleDoubleProperty(this, "currentRotation", 10);
		sizeing = new TextSizingUtility();
		
        final SteeringWheelGroup background = new SteeringWheelGroup(control.getInitialWidth(), currentRotation.doubleValue());
        background.setManaged(false); 
		Group g = new Group(background);
		

        final Point2D centerPoint = new Point2D(control.getInitialWidth()/2, control.getInitialWidth()/2);
        
        createAndAddLabels(g, centerPoint);
        
        background.setTranslateX(centerPoint.getX() - background.getImageWidth()/2);
		background.setTranslateY(centerPoint.getY() - background.getImageWidth()/2 - yTranslation);
		getChildren().add(g);
	}
	/**
	 * Create the labels and add them to the group.
	 * @param g group the labels are added to
	 * @param centerPoint central point around which the labels are placed.
	 */
	private void createAndAddLabels(Group g, final Point2D centerPoint) {
		List<String> values = control.valuesProperty();
        angles = new ArrayList<>(values.size());

		double angle = currentRotation.doubleValue() + 180;
        double step = (180.0 - 2 * currentRotation.doubleValue()) / (values.size() - 1);
        int radius = 100;
        for (String value : values) {
        	angles.add(angle);
			Point2D basePoint = calculateBasePoint(centerPoint, radius, angle);
            Dimension2D dim = sizeing.calculate(value, control.getFont());
            Label l = new Label(value);
            l.setFont(control.getFont());
            l.getStyleClass().add("openPatricianSlider");
            l.setTranslateX(basePoint.getX() - dim.getWidth() * calculateMoveFactor(angle));
            l.setTranslateY(basePoint.getY() - dim.getHeight());
            g.getChildren().add(l);
            angle += step;
        }
        currentRotation.setValue(control.getSelectedIndex() * step);
	}
	/**
	 * Snap to the correct angle. Correct angle is angle belonging to the nearest label.
	 */
	void snapToNearestAngle() {
		double currentAngle = currentRotation.doubleValue() + 180;
		double currentMin = 360;
		int minIndex = 0;
		for (int i = 0; i < angles.size(); i++) {
			double angle = angles.get(i);
			double diff = Math.abs(angle - currentAngle);
			if (diff < currentMin) {
				currentMin = diff;
				minIndex = i;
			}
		}
		Double destinationAngle = angles.get(minIndex);
		if (destinationAngle < 180 + 10 || destinationAngle > 360 - 10) {
			throw new IllegalStateException("Angle is out of range: "+currentRotation.doubleValue()+" -> "+destinationAngle);
		}
		control.selectedIndexProperty().set(minIndex);
		currentRotation.set(destinationAngle - 180);
	}
    /**
     * Calculate the angle between the vector horizontally to the left from the center
     * and the current point.
     * @param center point
     * @param point current point
     * @return angle in degree
     */
    double angleBetween2Lines(Point2D center, Point2D point) {
//System.out.println("center="+center+", point="+point);
    	double slope2 = calculateSlope(center, point);
        double angle = Math.atan(slope2);
        if (point.getX() > center.getX()) {
            angle = Math.PI - angle;
        }
//System.out.println("Slope: "+slope2+" angle "+Math.toDegrees(angle));
        return Math.abs(Math.toDegrees(angle));
    }
	/**
	 * Caluculate the slope of the line defined by two points.
	 * The first point is the center of a circle and the second
	 * point roughly lies on the circle.
	 * @param center point
	 * @param point on the circle
	 * @return slope of the connecting line.
	 */
	double calculateSlope(Point2D center, Point2D point) {
		double absSlope = Math.abs((point.getY() - center.getY()) / (point.getX() - center.getX()));
		if (point.getY() > center.getY()) {
			if (point.getX() > center.getX()) {
				// bottom right
				return -absSlope;
			} else {
				// bottom left
				return absSlope;
			}
		} else {
			if (point.getX() > center.getX()) {
				// top right
				return absSlope;
			} else {
				// top left
				return -absSlope;
			}
		}
	}

	/**
	 * Calculate the factor a label has to be moved based on the position on the curve.
	 * @param angle in degree in the range [0,180]
	 * @return move factor in the range [0,1]
	 */
    double calculateMoveFactor(double angle) {
        double deg = Math.toRadians(angle);
        double moveFactor = (Math.cos(deg - Math.PI) + 1) / 2;
        return moveFactor;
    }

    /**
     * Calculate the base point for the label. This is the point on the arc, matching the angle.
     * @param center point of the circle
     * @param radius radius of the circle
     * @param angle in degree in [0,180]
     * @return Point on the circle
     */
    Point2D calculateBasePoint(Point2D center, double radius,
            double angle) {
        float newX = (float) (center.getX() + radius * Math.cos(Math.toRadians(angle)));
        float newY = (float) (center.getY() + radius * Math.sin(Math.toRadians(angle)));
        return new Point2D(newX, newY);
    }
	
    /**
     * Class encapsulating the steering wheel image.
     * @author Andi Hotz, (c) Sahits GmbH, 2013
     * Created on Sep 29, 2013
     *
     */
    private class SteeringWheelGroup extends Group {
        public SteeringWheelGroup(int destinationWidth, double angle) {
            InputStream is = getClass().getResourceAsStream("SteeringWheel.png");
            Image unscaled = new Image(is);
            double scale = destinationWidth / unscaled.getWidth();
            ImageView steeringWheel = new ImageView(unscaled);
            steeringWheel.setScaleX(scale);
            steeringWheel.setScaleY(scale);
            steeringWheel.setRotate(angle);
            steeringWheel.rotateProperty().bind(currentRotation);
            Group rotationGroup = new Group(steeringWheel);
            float gapCausedByScale = (getImageWidth() - destinationWidth) / 2;
            final int radius = destinationWidth / 2;
            Rectangle clip = new Rectangle(gapCausedByScale, gapCausedByScale,
                    destinationWidth, radius);
            rotationGroup.setClip(clip);

            //final Point2D center = new Point2D(radius + gapCausedByScale, radius + gapCausedByScale - yTranslation);

            Circle circle = new Circle(radius, radius, radius, Color.RED);

            Circle innerCircle =     new Circle(radius, radius, radius * 0.6);

            Line line = new Line(0, radius, destinationWidth, radius);
            line.setStrokeWidth(10);
            line.setRotate(angle/2);

            Shape finalShape = Shape.subtract(circle, innerCircle);
            finalShape.setFill(Color.TRANSPARENT);
            finalShape.setTranslateX(gapCausedByScale);
            finalShape.setTranslateY(gapCausedByScale);
            

            finalShape.rotateProperty().bind(currentRotation);
            
            finalShape.setOnMouseDragged(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                	Bounds bounds = OpenPatricianSliderSkin.this.getBoundsInParent();
                	Point2D center = OpenPatricianSliderSkin.this.localToScene(bounds.getWidth()/2,bounds.getHeight());
    				if (mouseEvent.getSceneY() < center.getY()) {
	
	                    final Point2D draggedTo = new Point2D((float)mouseEvent.getSceneX(), (float)mouseEvent.getSceneY());
	                    double angle = angleBetween2Lines(center, draggedTo);
						if (angle < 10) {
							angle = 10;
						}
						if (angle > 170) {
							angle = 170;
						}
	                    currentRotation.set(angle);
    				}
                }
            });
            finalShape.setOnMouseReleased(new EventHandler<MouseEvent>() {

                @Override
                public void handle(MouseEvent event) {
                    snapToNearestAngle();
                }

            });

            rotationGroup.getChildren().add(finalShape);

            this.getChildren().add(rotationGroup);
        }

        public final int getImageWidth() {
            return 479;
        }
    }
    
}
