/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package ch.sahits.game.javafx.effects.transform;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.ObservableList;
import javafx.geometry.Point2D;
import javafx.scene.shape.Polygon;

/**
 * The perspective transformation can be expressed as a matrix multiplication:
 * <img src="http://upload.wikimedia.org/math/d/6/5/d65e582afbf418e4989971cdee42a167.png"/>.
 * The transformation matrix of a polygon is defined by the transformation of the bounding
 * box of the polygon. The general case where the four corners of the bounding box are
 * translated can be reduced to the special case where only three corners are translated.
 * Therefore this transformation algorithm will take the source and destination points
 * in 2D space of the three corners that are moved.
 * @author andi
 */
public class PolygonPerspectiveTransformation {
    
    private final static Translation2D ORIGIN_ORIGIN = new Translation2D(0, 0, 0, 0);
    
    private final ObjectProperty<Translation2D> point1 = new SimpleObjectProperty<>(this, "point-1",ORIGIN_ORIGIN);
    private final ObjectProperty<Translation2D> point2 = new SimpleObjectProperty<>(this, "point-2", ORIGIN_ORIGIN);
    private final ObjectProperty<Translation2D> point3 = new SimpleObjectProperty<>(this, "point-3", ORIGIN_ORIGIN);
    
    /* Initialize the Matrix with the identity */
    private double a11 = 1;
    private double a12 = 0;
    private double a21 = 0;
    private double a22 = 1;
    private double b1 = 0;
    private double b2 = 0;
    /**
     * Initializing the transormation by adding transformation listeners on the points.
     */
    public PolygonPerspectiveTransformation() {
        point1.addListener(new RecalculationEventListener());
        point2.addListener(new RecalculationEventListener());
        point3.addListener(new RecalculationEventListener());
   }
    /**
     * Recalculate the matrix values based on the points.
     */
    private void calculate() {
        a12 = calculateA12();
        a11 = calculateA11(); // requires recalculated a12
        b1 = calculateB1(); // requires recalculated a11 and a12
        a22 = calculateA22();
        a21 = calculateA21(); // requires recalculated a22
        b2 = calculateB2(); // requires recalculated a21 and a22
    }
    private double calculateA12() {
        double y1 = point1.getValue().getSource().getY();
        double y2 = point2.getValue().getSource().getY();
        double y3 = point3.getValue().getSource().getY();
        double x1 = point1.getValue().getSource().getX();
        double x2 = point2.getValue().getSource().getX();
        double x3 = point3.getValue().getSource().getX();
        double x1d = point1.getValue().getDestination().getX();
        double x2d = point2.getValue().getDestination().getX();
        double x3d = point3.getValue().getDestination().getX();
        
        double diff_x2_x1 = x2 - x1;
        double diff_x3_x1 = x3 - x1;
        
        double top = x3d - x1d + ((x1d - x2d)*diff_x3_x1/diff_x2_x1);
        double bottom = y3 - y1 - ((y2 - y1)*diff_x3_x1/diff_x2_x1);
        
        return top / bottom;
    }
    private double calculateA11() {
        double x1 = point1.getValue().getSource().getX();
        double x2 = point2.getValue().getSource().getX();
        double y1 = point1.getValue().getSource().getY();
        double y2 = point2.getValue().getSource().getY();
        double x1d = point1.getValue().getDestination().getX();
        double x2d = point2.getValue().getDestination().getX();
        
        double top = x2d - x1d - a12*(y2 - y1);
        double bottom = x2 - x1;
        
        return top / bottom;
   }
    private double calculateB1() {
        double x1 = point1.getValue().getSource().getX();
        double y1 = point1.getValue().getSource().getY();
        double x1d = point1.getValue().getDestination().getX();
        
        return x1d - a11*x1 - a12*y1;
   }
    
    private double calculateA22() {
        double y1 = point1.getValue().getSource().getY();
        double y2 = point2.getValue().getSource().getY();
        double y3 = point3.getValue().getSource().getY();
        double x1 = point1.getValue().getSource().getX();
        double x2 = point2.getValue().getSource().getX();
        double x3 = point3.getValue().getSource().getX();
        double y1d = point1.getValue().getDestination().getY();
        double y2d = point2.getValue().getDestination().getY();
        double y3d = point3.getValue().getDestination().getY();
        
        double diff_x2_x1 = x2 - x1;
        double diff_x3_x1 = x3 - x1;
        
        double top = y3d - y1d - ((y2d - y1d)*diff_x3_x1/diff_x2_x1);
        double bottom = y3 - y1 - ((y2 - y1)*diff_x3_x1/diff_x2_x1);
        
        return top / bottom;
    }
    private double calculateA21() {
        double y1 = point1.getValue().getSource().getY();
        double y2 = point2.getValue().getSource().getY();
        double x1 = point1.getValue().getSource().getX();
        double x2 = point2.getValue().getSource().getX();
        double y1d = point1.getValue().getDestination().getY();
        double y2d = point2.getValue().getDestination().getY();
        
        double top = y2d - y1d - a22*(y2 - y1);
        double bottom = x2 - x1;
        
        return top / bottom;
    }
    
    private double calculateB2() {
        double x1 = point1.getValue().getSource().getX();
        double y1 = point1.getValue().getSource().getY();
        double y1d = point1.getValue().getDestination().getY();
        
        return y1d - a21*x1 - a22*y1;
   }
    /**
     * Transform a polygon using the defined perspective transformation
     * @param polygon
     * @return 
     */
    public Polygon transform(Polygon polygon) {
        List<Point2D> points = convertToPoints(polygon.getPoints());
        Polygon transformedPolygon = new Polygon();
        for (Point2D p : points) {
            double xd = p.getX()*a11 + p.getY()*a12 + b1;
            double yd = p.getX()*a21 + p.getY()*a22 + b2;
            transformedPolygon.getPoints().addAll(xd, yd);
        }
        return transformedPolygon;
    }
 
    private List<Point2D> convertToPoints(ObservableList<Double> points) {
        ArrayList<Point2D> list = new ArrayList<>();
        for (Iterator<Double> it = points.iterator(); it.hasNext();) {
            double x = it.next();
            double y = it.next();
            list.add(new Point2D(x, y));
        }
        return list;
    }

    
    public Translation2D getPoint3() {
        return point3.get();
    }

    public void setPoint3(Translation2D value) {
        point3.set(value);
    }

    public ObjectProperty<Translation2D> point3Property() {
        return point3;
    }

    public Translation2D getPoint2() {
        return point2.get();
    }

    public void setPoint2(Translation2D value) {
        point2.set(value);
    }

    public ObjectProperty<Translation2D> point2Property() {
        return point2;
    }

    public Translation2D getPoint1() {
        return point1.get();
    }

    public void setPoint1(Translation2D value) {
        point1.set(value);
    }

    public ObjectProperty<Translation2D> point1Property() {
        return point1;
    }


    /**
     * Ensure that the values are recalculated on property change.
     */
    private class RecalculationEventListener implements ChangeListener<Translation2D> {
        @Override
        public void changed(ObservableValue<? extends Translation2D> ov, Translation2D t, Translation2D t1) {
            calculate();
        }
    }
    
    
    
}
