package ch.sahits.game.javafx.event;

import javafx.event.Event;
import javafx.event.EventHandler;

import java.util.ArrayList;
import java.util.List;

/**
 * Aggregation event handler that allows adding multiple event handlers for the same event type.
 * This facilitates creating event handlers for specific purposes.
 * The event handler are all called in the handle method in the order they are added.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Nov 08, 2014
 */
public class AggregateEventHandler<T extends Event> implements EventHandler<T> {

    private final List<EventHandler<T>> eventHandlers;

    public AggregateEventHandler() {
        eventHandlers = new ArrayList<>();
    }

    /**
     * Add a new event handler.
     * @param eventHandler to be added.
     */
    public void addEventHandler(EventHandler<T> eventHandler) {
        eventHandlers.add(eventHandler);
    }

    /**
     * Loop over all event handlers and delegate the event to each one of them.
     * If the event is consumed by one eventhandler the loop is broken.
     * @param t event.
     */
    @Override
    public void handle(T t) {
        for (EventHandler<T> eventHandler : eventHandlers) {
            if (!t.isConsumed()) {
                eventHandler.handle(t);
            } else {
                break;
            }
        }
    }
}
