package ch.sahits.game.javafx.control;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.geometry.Orientation;
import javafx.scene.Node;
import javafx.scene.control.Control;
import javafx.scene.layout.FlowPane;
import javafx.scene.text.Font;
import javafx.scene.text.Text;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Text control which can hold text mixed with controls.
 * The Decorated text is aimed to fill the whole horizontal space excessivly up to its
 * wrapping width.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 23, 2013
 */
@ClassCategory(EClassCategory.JAVAFX)
public class DecoratedText extends FlowPane implements IApplicableStyle, Styleable {

    private IntegerProperty wrappingWidth;
    private ReadOnlyObjectProperty<Font> injectedFont = null;

    /**
     * Contructor suppling an extarnal font to use.
     * @param font
     * @deprecated Use the constructor using the style class instead.
     */
    @Deprecated
    public DecoratedText(Font font) {
        super(Orientation.HORIZONTAL);
        wrappingWidth = new SimpleIntegerProperty(this, "wrappingWidth", 0);
        getStylesheets().add(getClass().getResource(getClass().getSimpleName()+".css").toExternalForm());
        this.injectedFont = new SimpleObjectProperty<>(this, "font", font);
        prefWidthProperty().bindBidirectional(wrappingWidth);
    }

    /**
     * Constructor using the base.css style sheet and the supplied style class to define the font.
     */
    public DecoratedText(String styleClass) {
        super(Orientation.HORIZONTAL);
        wrappingWidth = new SimpleIntegerProperty(this, "wrappingWidth", 0);
        getStylesheets().add(getClass().getResource(getClass().getSimpleName()+".css").toExternalForm());
        getStylesheets().add(this.getClass().getResource("/styles/base.css").toExternalForm());
        getStyleClass().add("styleClass");
        prefWidthProperty().bindBidirectional(wrappingWidth);
    }
    /**
     * Append text. If the text represents a paragraph (indicated by '\n'), it
     * is not broken up into its parts. To each part a space is added ' '.
     * @param text
     */
    public void append(String text) {
        Font chosenFont = choseFont();
        if (text.endsWith("\n")) {
            Text decoText = new Text(text);
            //decoText.getStyleClass().add("decoratedText-text");
            decoText.setFont(chosenFont);
            decoText.wrappingWidthProperty().bind(wrappingWidth);
            getChildren().add(decoText);
        } else {
            String[] parts = text.split(" ");
            for (String part : parts) {
                Text decoText = new Text(part+" ");
                //decoText.getStyleClass().add("decoratedText-text");
                decoText.setFont(chosenFont);
                getChildren().add(decoText);
            }

        }
    }
    private Font choseFont() {
        if (injectedFont != null) {
            return injectedFont.get();
        } else {
            return getFont();
        }
    }

    /**
     * Append a control.
     * @param control
     */
    public void append(Node control) {
         getChildren().add(control);
    }

    public int getWrappingWidth() {
        return wrappingWidth.get();
    }

    public IntegerProperty wrappingWidthProperty() {
        return wrappingWidth;
    }

    public void setWrappingWidth(int wrappingWidth) {
        this.wrappingWidth.set(wrappingWidth);
    }
    @Override
    public void applyStyle(String style) {
        for (Node node : getChildren()) {
           if (node instanceof IApplicableStyle) {
               ((IApplicableStyle)node).applyStyle(style);
           } else {
               node.setStyle(style);
           }
        }
    }

    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }
    public void setFont(Font font) {
        this.font.set(font);
    }
    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, DecoratedText.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData< DecoratedText, Font> FONT =
                new CssMetaData<DecoratedText, Font>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(DecoratedText control) {
                        return control.font == null || !control.font.isBound();
                    }
                    @Override
                    public StyleableProperty<Font> getStyleableProperty(DecoratedText control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;
        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }
    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }
    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }
}
