package ch.sahits.game.javafx.control;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.util.model.CenteredLine;
import ch.sahits.game.openpatrician.util.model.Heading;
import ch.sahits.game.openpatrician.util.model.ITextComponent;
import ch.sahits.game.openpatrician.util.model.Paragraph;
import ch.sahits.game.openpatrician.util.model.Text;
import com.google.common.base.Joiner;
import javafx.scene.text.Font;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

/**
 * Utility class to help separate the spliting up of a text into lines and pages.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Aug 7, 2013
 *
 */
@ClassCategory(EClassCategory.JAVAFX)
public class PageSplitter {
	public static final String CENTERED_STYLE = "pagination-centered";

	public static final String PARAGRAPH_STYLE = "pagination-paragraph";

	public static final String HEADING_STYLE = "pagination-heading";
	
	public static final String[] STYLES = {HEADING_STYLE, PARAGRAPH_STYLE, CENTERED_STYLE};


	private final double pageHeigth;
	private final double pageWidth;
	private final double lineSpacing;
	private final double initialHeigth;
	private final Font font;
	
	private TextSizingUtility sizing;
	private IWordSplitter splitter;
	/**
	 * Initialize the page splitter.
	 * @param pageHeigth height of a page
	 * @param pageWidth width of a page
	 * @param lineSpacing spacing between two lines
	 * @param initialHeigth initial height of the page
	 * @param font to be used.
	 */
	public PageSplitter(double pageHeigth, double pageWidth,
			double lineSpacing, double initialHeigth, Font font) {
		super();
		this.pageHeigth = pageHeigth;
		this.pageWidth = pageWidth;
		this.lineSpacing = lineSpacing;
		this.initialHeigth = initialHeigth;
		this.font = font;
		sizing = new TextSizingUtility();
	}
	/**
	 * Override the sizing utility for test purposes.
	 * @param sizing
	 */
	void overrideTextSizingUtility(TextSizingUtility sizing) {
		this.sizing = sizing;
	}
	public void setWordSplitter(IWordSplitter splitter) {
		this.splitter = splitter;
	}
	/**
	 * Splitting up the text onto pages.
	 * @param text to be split
	 * @return list of Pages.
	 */
	public List<Page> splitIntoPages(Text text) {
		return splitIntoPages(text, false);
	}
	/**
	 * Splitting up the text onto pages.
	 * Using the additional flag you can control the behavior on line setup.
	 * This becomes important on recursive calls, where a word was splitt and
	 * should be reassambled on a single line. 
	 * @param text to be split
	 * @param eliminateWhiteSpaces flag indicating if the words on a line should be concatenated (remove spaces)
	 * @return list of Pages.
	 */
	private List<Page> splitIntoPages(Text text, boolean eliminateWhiteSpaces) {
System.out.println("Eleminate whitespaces: "+eliminateWhiteSpaces);		
		List<Page> pages = new ArrayList<Page>();
    	List<ITextComponent> components = text.getText();
    	StringBuilder oneLine = new StringBuilder();
    	Page page = new Page();
    	double currentPageHeight = initialHeigth;
    	for (Iterator<ITextComponent> iterator = components.iterator(); iterator.hasNext();) {
			ITextComponent component = iterator.next();
    		String s;
    		String textStyle = "";
			if (component instanceof Heading) {
				s = ((Heading) component).getHeader();
				textStyle = HEADING_STYLE;
			} else if (component instanceof Paragraph) {
				s = ((Paragraph) component).getParagraph();
				textStyle = PARAGRAPH_STYLE;
			} else if (component instanceof CenteredLine) {
				s = ((CenteredLine)component).getLine();
				textStyle = CENTERED_STYLE;
			} else {
				throw new IllegalStateException("Unhandled text component");
			}
    		StringTokenizer tokenizer = new StringTokenizer(s);
    		while (tokenizer.hasMoreTokens()) {
    			String word = tokenizer.nextToken();
    			if (oneLine.length() > 0) {
    				oneLine.append(" ");
    			}
    			oneLine.append(word);
    			double lineWidth = sizing.computeTextWidth(font, oneLine.toString(), Double.MAX_VALUE);
    			// line length exceeds available space
    			boolean lineAdded = false; // flag to avoid adding contents twice, when it has to be split
    			if (isLineTooLong(lineWidth)) {
    				String line;
    				if (oneLine.lastIndexOf(" ") > 0) {
    					line = oneLine.substring(0, oneLine.lastIndexOf(" "));
    				} else {
    					line = oneLine.toString();
    				}
    				Line l = new Line();
					if (eliminateWhiteSpaces) {
						line = line.replaceAll(" ", "");
					}
    				l.setLine(line);
    				l.setStyleClass(textStyle);
    				if (tokenizer.hasMoreTokens()) { // otherwise it was already added as the last word
    					page.addLine(l);
    				}
    				lineWidth = sizing.computeTextWidth(font, word, Double.MAX_VALUE);
    				if (isLineTooLong(lineWidth)) {
    					if (!tokenizer.hasMoreTokens()) {
        					page.addLine(l);
    					}
    					String[] parts = splitter.split(word);
    					Text tmpText = new Text();
    					Paragraph p = new Paragraph(Joiner.on(' ').join(parts));
    					tmpText.addParagraph(p);
    					List<Page> tmpPages = splitIntoPages(tmpText, true);
    					if (tmpPages.size() > 1) {
    						throw new IllegalStateException("Got a more than on page out of a single word.");
    					}
    					List<Line> lines = tmpPages.get(0).getLines();
    					int index = 0;
    					while(index < lines.size()) {
    						l = lines.get(index++);
    						String compacted = l.getLine().replaceAll(" ", "");
    						l.setLine(compacted);
    	    				currentPageHeight = updatePageHeight(compacted, currentPageHeight);
    	    				if (currentPageHeight > pageHeigth) {
    	    					// line on the next page
    	        				pages.add(page);
    	        				currentPageHeight = initialHeigth;
    	        				page = new Page();
    	    				}
	        				l.setStyleClass(textStyle);
	        				page.addLine(l);
    					}
    					oneLine = new StringBuilder();
    					lineAdded = true;
    				} else {
    					oneLine = new StringBuilder(word);
    					lineAdded = false;
    				}
    				currentPageHeight = updatePageHeight(line, currentPageHeight);
    			}
    			// last word
    			if (!tokenizer.hasMoreTokens() && !lineAdded) {
    				Line l = new Line();
    				String line = oneLine.toString();
					if (eliminateWhiteSpaces) {
						line = line.replaceAll(" ", "");
					}
    				l.setLine(line);
    				l.setStyleClass(textStyle);
    				page.addLine(l);
    				currentPageHeight = updatePageHeight(oneLine.toString(), currentPageHeight);
    				oneLine = new StringBuilder(); // clear the cache
    			}
    			// Last line just added cannot be displayed on the page
    			if (currentPageHeight > this.pageHeigth) {
    				Line lastLine = page.removeLastLine();
    				pages.add(page);
    				currentPageHeight = initialHeigth;
    				page = new Page();
    				page.addLine(lastLine);
    				currentPageHeight = updatePageHeight(lastLine.getLine(), currentPageHeight);
    			}
    		} 
    		// all words from the text component
    		if (!iterator.hasNext()) {
    			pages.add(page); // last page
    		}
		}
System.out.println("Leave splittIntoPages");		
    	return pages;
    }
	private boolean isLineTooLong(double lineWidth) {
		return lineWidth > pageWidth;
	}
	/**
	 * Compute the line height and add it to the page height.
	 * @param line to be checked
	 * @param currentPageHeight current page height without <code>line</code>
	 * @return new page height
	 */
	private double updatePageHeight(String line, double currentPageHeight) {
		double lineHeigth = sizing.computeTextHeight(font, line, Double.MAX_VALUE);
		currentPageHeight += lineSpacing + lineHeigth;
		return currentPageHeight;
	}
	
	
	
}
