package ch.sahits.game.javafx.control.skin;

import ch.sahits.game.javafx.control.Line;
import ch.sahits.game.javafx.control.NonAlphaNumWordSplitter;
import ch.sahits.game.javafx.control.Page;
import ch.sahits.game.javafx.control.PageSplitter;
import ch.sahits.game.javafx.control.PaginationV2;
import ch.sahits.game.javafx.control.PlaceHolder;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.util.model.Text;
import javafx.beans.binding.DoubleBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.geometry.Insets;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.control.SkinBase;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.AnchorPane;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;

import java.util.ArrayList;
import java.util.List;


/**
 * Skin implementation for the Pagination control.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Aug 4, 2013
 */
@ClassCategory(EClassCategory.JAVAFX)
public class PaginationSkinV2 extends SkinBase<PaginationV2> {
	
	private static final double DROP_SHADOW_PADDING = 10;

    private final Label backLbl;
    private final Label nextLbl;
    private final VBox contentContainer;
    @ListType(Page.class)
    private final List<Page> pages;
    private int currentPage;
    private final DoubleProperty contentHeight;
    private final DoubleProperty contentWidth;
    private final BooleanProperty backButtonVisibility;
    private final BooleanProperty nextButtonVisibility;
    
    private final EventHandler<MouseEvent> backButtonEventHandler;
    private final EventHandler<MouseEvent> nextButtonEventHandler;
    
    private final PaginationV2 control;
    
    // TODO aho Sep 15, 2013: as everything is readonly set it instead of binding
    
    public PaginationSkinV2(final PaginationV2 control) {
        super(control);
        this.control = control;
        control.getStyleClass().add("paginationBG");
        // Initialize the internals
        pages = new ArrayList<>();
        currentPage = 0;
        backButtonVisibility = new SimpleBooleanProperty(this, "backButtonVisibility", control.getFirstLastPageAction() != null);
        nextButtonVisibility = new SimpleBooleanProperty(this, "nextButtonVisibility", true);
        backButtonEventHandler = new BackButtonEventHandler();
        nextButtonEventHandler = new NextButtonEventHanlder();

        // Initialize the controls
        Insets defaultInsets = new Insets(10);
        backLbl = new Label();
        backLbl.setId(getClass().getName() + "backButton");
        backLbl.fontProperty().bind(control.navigationLabelFontProperty());
        backLbl.textProperty().bind(control.backButtonLabelProperty());
        backLbl.visibleProperty().bind(backButtonVisibility);
        backLbl.setPadding(defaultInsets);
        nextLbl = new Label();
        nextLbl.setId(getClass().getName() + "nextButton");
        nextLbl.fontProperty().bind(control.navigationLabelFontProperty());
        nextLbl.textProperty().bind(control.nextButtonLabelProperty());
        nextLbl.visibleProperty().bind(nextButtonVisibility);
        nextLbl.setPadding(defaultInsets);
        contentContainer = new VBox();
        contentContainer.getStyleClass().add("containerVBox");
        contentContainer.setMaxWidth(control.getContentMaxWidth());
        contentContainer.setPadding(defaultInsets);

        
        contentHeight = new SimpleDoubleProperty(this, "contentHeight", 0.0);
        contentHeight.bind(control.contentMaxHeightProperty());
        contentWidth = new SimpleDoubleProperty(this, "contentWidth", 0.0);
        DoubleBinding contentWidthBinding = control.contentMaxWidthProperty()
        		.subtract(control.paddingLocalProperty().getValue().getLeft())
        		.subtract(control.paddingLocalProperty().getValue().getRight())
        		.subtract(2*DROP_SHADOW_PADDING);
		contentWidth.bind(contentWidthBinding);
        control.paddingLocalProperty().addListener(new ChangeListener<Insets>(){

			@Override
			public void changed(ObservableValue<? extends Insets> observable,
					Insets oldValue, Insets newValue) {
		        double sideInsets = newValue.getLeft() + newValue.getRight();
		        contentWidth.set(control.contentMaxWidthProperty().subtract(sideInsets).doubleValue());
				
			}
        	
        });
        
        AnchorPane buttonRow = new AnchorPane();
        buttonRow.getChildren().addAll(backLbl, nextLbl);
        buttonRow.getStyleClass().add("buttonRow");
        AnchorPane.setLeftAnchor(backLbl, 0.0);
        AnchorPane.setRightAnchor(nextLbl, control.getPaddingLocal().getRight());
        final BorderPane overAllLayout = new  BorderPane();
        PlaceHolder placeHolder = new PlaceHolder(1, contentHeight.doubleValue());
        placeHolder.heightLocalProperty().bind(contentHeight);
        overAllLayout.setPadding(new Insets(DROP_SHADOW_PADDING));
        overAllLayout.getStyleClass().add("overAllLayoutAnchorPane");
        overAllLayout.setLeft(contentContainer);
        overAllLayout.setCenter(placeHolder);
        overAllLayout.setBottom(buttonRow);
        buttonRow.prefWidthProperty().bind(overAllLayout.widthProperty());

        overAllLayout.maxWidthProperty().bind(contentWidthBinding);
        
        
        initialalizeNewText(control.getText());
        
        Group group = new Group(overAllLayout);
        group.setManaged(false);
        group.setLayoutX(control.paddingLocalProperty().getValue().getLeft());
        
        getChildren().add(group);
        
        PaginationLabelHoverChangeListener changeListenerBack = new PaginationLabelHoverChangeListener(backLbl); 
        PaginationLabelHoverChangeListener changeListenerNext = new PaginationLabelHoverChangeListener(nextLbl); 
       if (control.getBackButtonNode() != null && control.getNextButtonNode() != null) {
            control.getBackButtonNode().setOnMouseClicked(backButtonEventHandler);
            control.getNextButtonNode().setOnMouseClicked(nextButtonEventHandler);
            control.getBackButtonNode().hoverProperty().addListener(changeListenerBack);
            control.getNextButtonNode().hoverProperty().addListener(changeListenerNext);
        } else {
            nextLbl.setOnMouseClicked(nextButtonEventHandler);
            backLbl.setOnMouseClicked(backButtonEventHandler);
            backLbl.hoverProperty().addListener(changeListenerBack);
            nextLbl.hoverProperty().addListener(changeListenerNext);
        }
    }
    /**
     * Refresh the content to be displayed on the current page.
     */
    private void updateDisplay() {
        clearContents();
        Page page = pages.get(currentPage);
        for (Line line : page.getLines()) {
            final Label l = new Label(line.getLine());
            l.getStyleClass().add(line.getStyleClass());
            if (line.getStyleClass().equals(PageSplitter.CENTERED_STYLE)){
            	l.setFont(control.getCenteredFont());
            	BorderPane centered = new BorderPane();
            	centered.setCenter(l);
            	contentContainer.getChildren().add(centered);
            } else {
            	if (line.getStyleClass().equals(PageSplitter.PARAGRAPH_STYLE)) {
            		l.setFont(control.getParagraphFont());
            	} else if (line.getStyleClass().equals(PageSplitter.HEADING_STYLE)) {
            		l.setFont(control.getHeaderFont());
            	} else {
            		throw new IllegalStateException("Unknown style for font setting");
            	}
            	contentContainer.getChildren().add(l);
            }
        }
    }
    /**
     * Clear the current contents.
     */
    private void clearContents() {
    	contentContainer.getChildren().removeAll(contentContainer.getChildren());
    }
    /**
     * Split up the text into lines and pages, that can be displayed.
     * @param text 
     */
    private void splitTextToPages(Text text) {
		final double width = getContentWidth();
		final double heigh = contentHeight.doubleValue();
		Font font = control.getNavigationLabelFont();
    	PageSplitter splitter = new PageSplitter(heigh, width, contentContainer.getSpacing(), calculateContentInitialHeigth(), font);
    	splitter.setWordSplitter(new NonAlphaNumWordSplitter());
    	pages.addAll(splitter.splitIntoPages(text));

    }
	private double getContentWidth() {
		return contentWidth.get() - 2*DROP_SHADOW_PADDING;
	}
    /**
     * Calculate the initial height of the content without content.
     * @return 
     */
    private double calculateContentInitialHeigth() {
        Insets insets = contentContainer.getInsets();
        double top = snapSpace(insets.getTop());
        double bottom = snapSpace(insets.getBottom());
        return top+bottom;
    }
    /**
     * Initialize new text.
     * @param text 
     */
    private void initialalizeNewText(Text text) {
                pages.clear(); // Avoid duplicates
                splitTextToPages(text);
                currentPage = 0;
                updateDisplay();
    }
    /**
     * Event handler for the back button.
     */
    private class BackButtonEventHandler implements EventHandler<MouseEvent> {
            @Override
            public void handle(MouseEvent event) {
                if (currentPage > 0) {
                    currentPage--;
                    // display last page
                    updateDisplay();
                    nextButtonVisibility.set(true);
                    if (currentPage == 0 && control.getFirstLastPageAction() == null) {
                        backButtonVisibility.set(false);
                    }
                } else if (currentPage == 0) {
                	control.getFirstLastPageAction().handle(event); // delegate
                }
            }
    }
    /**
     * Event handler for the next button.
     */
    private class NextButtonEventHanlder implements EventHandler<MouseEvent> {
            @Override
            public void handle(MouseEvent event) {
                if (currentPage < pages.size()-1) {
                    currentPage++;
                    // display next page
                    updateDisplay();
                    backButtonVisibility.set(true);
                    if (currentPage == pages.size()-1 && control.getFirstLastPageAction() == null) {
                        nextButtonVisibility.set(false);
                    }
                } else if (currentPage == pages.size()-1) {
                	control.getFirstLastPageAction().handle(event); // delegate
                }
            }        
    }
}