package ch.sahits.game.javafx.service;

import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.event.WrappedDialogType;
import ch.sahits.game.javafx.model.EDialogType;
import ch.sahits.game.javafx.model.ENoticeBoardType;
import ch.sahits.game.javafx.model.NoticeBoardMenuEntry;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.service.ShipService;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.event.EventHandler;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.MessageSource;

import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;

/**
 * Provide the menu items for the ship selection notice board menu.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 19, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class NoticeBoardShipSelectionMenuProvider implements INoticeBoardMenuProvider {
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ShipService shipService;
    @Autowired
    protected DecoratedTextFactory textFactory;
    @Autowired
    private ApplicationContext context;
    @Override
    public LinkedHashSet<NoticeBoardMenuEntry> createMenuEntries(ICityPlayerProxyJFX proxy, ENoticeBoardType type) {
        LinkedHashSet<NoticeBoardMenuEntry> entries = new LinkedHashSet<>();

        List<INavigableVessel> vessels = proxy.getPlayersNavalVessels();
        for (INavigableVessel vessel : vessels) {
            int weaponsStrength = shipService.calculateShipsWeaponsStrength(vessel);
            if (vessel instanceof IShip) {
                Object[] args = new Object[]{EDialogType.SHIP_CONVOY};
                final Runnable runnable1 = (Runnable) context.getBean("viewChangeActionRunnable", args);
                EventHandler<MouseEvent> openDialogEvenHandler = evt -> {
                    proxy.activateShip(vessel);
                    if (!evt.getButton().equals(MouseButton.PRIMARY)) {
                        runnable1.run();
                    }
                };
                boolean disabled = false;
                final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.ship";
                NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, vessel, disabled, key,false);
                entry.setOnMouseSelect(openDialogEvenHandler);
                entries.add(entry);
            } else {
                // it is a convoy
                IConvoy convoy = (IConvoy) vessel;
                boolean disabled = !convoy.getOwner().equals(proxy.getPlayer());
                Object[] args = new Object[]{EDialogType.SHIP_CONVOY};
                final Runnable runnable1 = (Runnable) context.getBean("viewChangeActionRunnable", args);
                EventHandler<MouseEvent> openDialogConvoyEvenHandler = evt -> {
                    proxy.activateShip(vessel);
                    if (!evt.getButton().equals(MouseButton.PRIMARY)) {
                        runnable1.run();
                    }
                };
                    if (weaponsStrength < 5) {
                        final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.convoy.captain.smallstrangth.capacity";
                        NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, convoy, disabled, key, false);

                        entry.setOnMouseSelect(openDialogConvoyEvenHandler);
                        entries.add(entry);
                    } else {
                        final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.convoy.captain.bigstrangth.capacity";
                        NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, convoy, disabled, key, false);
                        entry.setOnMouseSelect(openDialogConvoyEvenHandler);
                        entries.add(entry);
                    }
                List<IShip> convoyMembers = convoy.getShips();
                for (IShip convoyMember : convoyMembers) {
                    disabled = !convoyMember.getOwner().equals(proxy.getPlayer());
                    final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.shipinconvoy";
                    NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, convoyMember, disabled, key, true);
                    if (!convoyMember.equals(convoy.getOrlegShip())) {
                        WrappedDialogType wrapper = new WrappedDialogType(EDialogType.LEAVE_CONVOY, convoy, convoyMember);
                        final Runnable run = (Runnable) context.getBean("viewChangeWrappedRunnable", wrapper);
                        entry.setOnMouseSelect(event -> {
                            proxy.activateShip(vessel);
                            if (!event.getButton().equals(MouseButton.PRIMARY)) {
                                run.run();
                            }
                        });
                    } else {
                        entry.setOnMouseSelect((evt) -> proxy.activateShip(vessel));
                    }
                    entries.add(entry);
                }
            } // end else convoy
        }
        return entries;
    }

    private NoticeBoardMenuEntry createNoticeBoardEntry(ICityPlayerProxyJFX proxy, INavigableVessel vessel, boolean disabled, String key, boolean inset) {
        String loadedText = messageSource.getMessage(key,
                new Object[]{vessel.getName(), vessel.getCapacity()}, locale.getCurrentLocal());
        return createMenuEntry(proxy, vessel, loadedText, disabled);
    }

    private NoticeBoardMenuEntry createMenuEntry(ICityPlayerProxyJFX proxy, INavigableVessel vessel, String loadedText, boolean disabled) {
        DecoratedText dt = textFactory.createDecoratedText(loadedText, new HashMap<>());
        NoticeBoardMenuEntry entry = new NoticeBoardMenuEntry(null);
        entry.setLabel(dt);
        entry.setDisabled(disabled);
        entry.setSelected(vessel.equals(proxy.getActiveShip()));
        return entry;
    }


}
