package ch.sahits.game.javafx.control;

import ch.sahits.game.javafx.model.NoticeBoardMenu;
import com.google.common.base.Preconditions;
import javafx.application.Platform;
import javafx.scene.Group;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.ScrollPane.ScrollBarPolicy;

import java.net.URL;
import java.util.LinkedList;
import java.util.Optional;

/**
 * Control representing the main menu.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 1, 2013
 *
 */
public abstract class NoticeBoard extends Group {

	public static final int DEFAULT_WIDTH = 216;
    private final NoticeBoardScrollPaneContent nbScrollPane;

	protected LinkedList<NoticeBoardMenu> noticeBoardMenu = new LinkedList<>();

	public NoticeBoard() {
		setManaged(false);
        String cssFileName = NoticeBoard.class.getSimpleName()+".css";
		URL resource = NoticeBoard.class.getResource(cssFileName);
		getStylesheets().add(resource.toExternalForm());
		getStyleClass().add("noticeBoardText");

		ScrollPane scrollPane = new ScrollPane();
		scrollPane.setLayoutX(20);
		scrollPane.setLayoutY(5);
		scrollPane.setMaxWidth(DEFAULT_WIDTH);
		scrollPane.setMinWidth(216);
		scrollPane.setMaxHeight(233);
		scrollPane.setMinHeight(233);
		scrollPane.setHbarPolicy(ScrollBarPolicy.NEVER);
		scrollPane.setVbarPolicy(ScrollBarPolicy.AS_NEEDED);
        nbScrollPane = new NoticeBoardScrollPaneContent();
        scrollPane.setContent(nbScrollPane);
		getChildren().addAll(scrollPane);
	}

    public void addMenu(NoticeBoardMenu menu) {
        if (noticeBoardMenu.contains(menu)) {
            throw new IllegalArgumentException("The menu may not be added twice.");
        }
        noticeBoardMenu.add(menu);
        if (Platform.isFxApplicationThread()) {
            nbScrollPane.setContent(Optional.of(menu));
        } else {
            Platform.runLater(() -> nbScrollPane.setContent(Optional.of(menu)));
        }
    }

    public void closeNoticeBoard() {
        if (!noticeBoardMenu.isEmpty()) {
            noticeBoardMenu.removeLast();
        }
        final Optional<NoticeBoardMenu> optMenu;
        if (noticeBoardMenu.isEmpty()) {
            optMenu = Optional.empty();
        } else {
            optMenu = Optional.of(noticeBoardMenu.getLast());
        }
        if (Platform.isFxApplicationThread()) {
            nbScrollPane.setContent(optMenu);
        } else {
            Platform.runLater(() -> nbScrollPane.setContent(optMenu));
        }
    }

    public void reset(NoticeBoardMenu menu) {
       NoticeBoardMenu last = noticeBoardMenu.getLast();
        Preconditions.checkArgument(menu.getType() == last.getType(), "The displayed menu has a different type than the one to replace it with.");
        noticeBoardMenu.removeLast();
        addMenu(menu);
    }


	
}
