package ch.sahits.game.javafx.service;

import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.DisplayMessageOverlayLabel;
import ch.sahits.game.javafx.control.NoticeBoard;
import ch.sahits.game.javafx.control.TextSizingUtility;
import ch.sahits.game.javafx.model.DisplayMessages;
import ch.sahits.game.javafx.model.ENoticeBoardType;
import ch.sahits.game.javafx.model.NoticeBoardMenuEntry;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.javafx.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.collections.ObservableList;
import javafx.event.EventHandler;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.StackPane;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.MessageSource;

import java.util.Iterator;
import java.util.LinkedHashSet;

/**
 * Provide the menu items for the messages in the notice board menu.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 19, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class NoticeBoardMessageMenuProvider implements INoticeBoardMenuProvider {
    @Autowired
    private DisplayMessages displayMessages;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private TextEllipsisCalculator ellipsisCalculator;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private TextSizingUtility sizeing;
    @Autowired
    private ApplicationContext context;
    @Override
    public LinkedHashSet<NoticeBoardMenuEntry> createMenuEntries(ICityPlayerProxyJFX proxy, ENoticeBoardType type) {
        ObservableList<DisplayMessage> messages = displayMessages.messagesProperty();
        double maxWidth = NoticeBoard.DEFAULT_WIDTH - 30; // scrollbar
        LinkedHashSet<NoticeBoardMenuEntry> entries = new LinkedHashSet<>();
        for (DisplayMessage message : messages) {
            Object[] args = message.getMessageArguments();
            for (int i = 0; i < args.length; i++) {
                if (args[i] instanceof EWare) {
                    args[i] = translator.getLocalDisplayName((EWare) args[i]);
                }
            }
            String translatedMessage = messageSource.getMessage(message.getMessageKey(), args, locale.getCurrentLocal());
            DecoratedText text = new DecoratedText();
            String messageString = ellipsisCalculator.calculateEllipsis(text.getFont(), translatedMessage, maxWidth);
            text = textFactory.createSimpleDecoratedText(messageString);
            EventHandler<MouseEvent> onMouseFocus = getDisplaMessageMouseEnterFocusEventHandler(text, translatedMessage);
            EventHandler<MouseEvent> onMouseExitFocus = getDisplayMessageMouseExitFocusEventHandler(text);
            text.setOnMouseEntered(onMouseFocus);
            text.setOnMouseExited(onMouseExitFocus);
            NoticeBoardMenuEntry entry = new NoticeBoardMenuEntry(null);
            entry.setLabel(text);
            if (message.hasDialogTemplate()) {
                args = new Object[]{message.getDialogTemplate()};
                Runnable runnable = (Runnable) context.getBean("viewChangeDialogTemplateRunnable", args);
                EventHandler<MouseEvent> openDialogEvenHandler = evt -> runnable.run();
                entry.setOnMouseSelect(openDialogEvenHandler);
            }
            entries.add(entry);
        }
        return entries;
    }

    private EventHandler<MouseEvent> getDisplaMessageMouseEnterFocusEventHandler(DecoratedText text, String message) {
        return event -> {
            final Group viewContainer = (Group) ((StackPane)text.getScene().getRoot()).getChildren().get(1);
            DisplayMessageOverlayLabel label = new DisplayMessageOverlayLabel(message, sizeing);
            double y = event.getSceneY();
            label.setLayoutY(y);
            label.setLayoutX(350);	// todo: andi 3/8/15: Use some kind of constant. Also consider that the width of the control board may be scalable at one time in the future
            viewContainer.getChildren().add(label);  // Add it to the scene and not the scroll pane and adjust the location
        };
    }
    private EventHandler<MouseEvent> getDisplayMessageMouseExitFocusEventHandler(DecoratedText text){
        return event -> {
            final Group viewContainer = (Group) ((StackPane)text.getScene().getRoot()).getChildren().get(1);
            for (Iterator<Node> iterator = viewContainer.getChildren().iterator(); iterator.hasNext(); ) {
                Node node =  iterator.next();
                if (node instanceof DisplayMessageOverlayLabel) {
                    iterator.remove();
                    break;
                }
            }
        };
    }
}
