package ch.sahits.game.javafx.control;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.util.model.Text;
import javafx.beans.property.*;
import javafx.event.EventHandler;
import javafx.geometry.Insets;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.paint.Color;
import javafx.scene.shape.Polygon;
import javafx.scene.text.Font;
import javafx.util.Builder;
/**
 * Builder for the PaginationV2 control.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Sep 14, 2013
 *
 */
@ClassCategory(EClassCategory.JAVAFX)
public class PaginationV2Builder implements Builder<PaginationV2> {

	private DoubleProperty contentMaxWidth;
	private DoubleProperty contentMaxHeight;
    @ObjectPropertyType(Insets.class)
	private ObjectProperty<Insets> paddingLocal;
	private StringProperty backButtonLabel;
	private StringProperty nextButtonLabel;
	@ObjectPropertyType(Text.class)
	private ObjectProperty<Text> text;
	private ReadOnlyObjectProperty<Font> navigationLabelFont;
	private ReadOnlyObjectProperty<Font> headerFont;
	private ReadOnlyObjectProperty<Font> centeredFont;
	private ReadOnlyObjectProperty<Font> paragraphFont;

	// optional
    @ObjectPropertyType(Node.class)
	private ObjectProperty<Node> backButtonNode;
    @ObjectPropertyType(Node.class)
	private ObjectProperty<Node> nextButtonNode;
    @ObjectPropertyType(EventHandler.class)
	private ObjectProperty<EventHandler<MouseEvent>> firstLastPageAction;

    private boolean withPolygonShading = false;

	private PaginationV2Builder() {
		firstLastPageAction(null);
		nextButtonNode(null);
		backButtonNode(null);
        if (System.getProperty("javafx.polygon.shading") != null) {
            withPolygonShading = true;
        }
	}

	public static PaginationV2Builder create() {
		return new PaginationV2Builder();
	}


	/**
	 * Define the max width for the content.
	 * 
	 * @param width
	 * @return
	 */
	public PaginationV2Builder contentMaxWidth(double width) {
		contentMaxWidth = new SimpleDoubleProperty(this, "contentMaxWidth",
				width);
		return this;
	}

	/**
	 * Define the max width for the content. The value of the property is taken
	 * as width
	 * 
	 * @param width
	 * @return
	 */
	public PaginationV2Builder contentMaxWidth(DoubleProperty width) {
		contentMaxWidth = new SimpleDoubleProperty(this, "contentMaxWidth",
				width.doubleValue());
		return this;
	}


	/**
	 * Define the max width for the content.
	 * 
	 * @param height
	 * @return
	 */
	public PaginationV2Builder contentMaxHeight(double height) {
		contentMaxHeight = new SimpleDoubleProperty(this, "contentMaxWidth",
				height);
		return this;
	}

	/**
	 * Define the max width for the content. The value of the property is taken
	 * as width
	 * 
	 * @param height
	 * @return
	 */
	public PaginationV2Builder contentMaxHeight(DoubleProperty height) {
		contentMaxHeight = new SimpleDoubleProperty(this, "contentMaxHeight",
				height.doubleValue());
		return this;
	}

	public PaginationV2Builder padding(Insets insets) {
		paddingLocal = new ObjectPropertyBase<Insets>(insets) {

			@Override
			public Object getBean() {
				return this;
			}

			@Override
			public String getName() {
				return "paddingLocal";
			}

		};
		return this;
	}

	public PaginationV2Builder backButtonLabel(String backButtonLabel) {
		this.backButtonLabel = new SimpleStringProperty(this,
				"backButtonLabel", backButtonLabel);
		return this;
	}

	public PaginationV2Builder nextButtonLabel(String nextButtonLabel) {
		this.nextButtonLabel = new SimpleStringProperty(this,
				"nextButtonLabel", nextButtonLabel);
		return this;
	}

	public PaginationV2Builder text(Text text) {
		this.text = new ObjectPropertyBase<Text>(text) {

			@Override
			public Object getBean() {
				return this;
			}

			@Override
			public String getName() {
				return "text";
			}
		};
		return this;
	}

	public PaginationV2Builder navigationLabelFont(final Font font) {
		navigationLabelFont = new ReadOnlyObjectPropertyBase<Font>() {

			@Override
			public Object getBean() {
				return this;
			}

			@Override
			public String getName() {
				return "navigationLabelFont";
			}

			@Override
			public Font get() {
				return font;
			}
		};
		return this;
	}

	public PaginationV2Builder headerFont(final Font font) {
		headerFont = new ReadOnlyObjectPropertyBase<Font>() {

			@Override
			public Object getBean() {
				return this;
			}

			@Override
			public String getName() {
				return "headerFont";
			}

			@Override
			public Font get() {
				return font;
			}
		};
		return this;
	}

	public PaginationV2Builder paragraphFont(final Font font) {
		paragraphFont = new ReadOnlyObjectPropertyBase<Font>() {

			@Override
			public Object getBean() {
				return this;
			}

			@Override
			public String getName() {
				return "paragraphFont";
			}

			@Override
			public Font get() {
				return font;
			}
		};
		return this;
	}

	public PaginationV2Builder centeredFont(final Font font) {
		centeredFont = new ReadOnlyObjectPropertyBase<Font>() {

			@Override
			public Object getBean() {
				return this;
			}

			@Override
			public String getName() {
				return "centeredFont";
			}

			@Override
			public Font get() {
				return font;
			}
		};
		return this;
	}

	public PaginationV2Builder backButtonNode(Node backButtonNode) {
        if (backButtonNode instanceof Polygon && withPolygonShading) {
            ((Polygon)backButtonNode).setFill(Color.DARKGREEN);
            backButtonNode.setOpacity(0.5);
        }
        this.backButtonNode = new SimpleObjectProperty<>(this, "backButtonNode", backButtonNode);
		return this;
	}

	public PaginationV2Builder nextButtonNode(Node nextButtonNode) {
        if (nextButtonNode instanceof Polygon && withPolygonShading) {
            ((Polygon)nextButtonNode).setFill(Color.DARKGREEN);
            nextButtonNode.setOpacity(0.5);
        }
		this.nextButtonNode = new SimpleObjectProperty<>(this, "nextButtonNode", nextButtonNode);
		return this;
	}

	public PaginationV2Builder firstLastPageAction(
			EventHandler<MouseEvent> firstLastPageAction) {
		this.firstLastPageAction =  new SimpleObjectProperty<>(this, "firstLastPageEventActionHandler", firstLastPageAction);
		return this;
	}

	@Override
	public PaginationV2 build() {
		return new PaginationV2(backButtonLabel, nextButtonLabel,
				backButtonNode, nextButtonNode, firstLastPageAction, text,
				contentMaxWidth, contentMaxHeight, paddingLocal,
				navigationLabelFont, headerFont, paragraphFont, centeredFont);
	}
}
