package ch.sahits.game.javafx.control.skin;

import ch.sahits.game.javafx.control.EventPlayerFrame;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import com.google.common.base.Preconditions;
import javafx.beans.property.ObjectProperty;
import javafx.geometry.Bounds;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import javafx.scene.Group;
import javafx.scene.control.SkinBase;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;

import java.io.InputStream;

/**
 * Skin for EventPlayerFrame.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 *         Created on Jan 03, 2017
 */
@ClassCategory(EClassCategory.JAVAFX)
public class EventPlayerFrameSkin extends SkinBase<EventPlayerFrame> {

    private final int FRAME_BORDER_LENGTH = 128;
    private final double SCALE_FACTOR = 0.6;  // 60%
    private final int FULL_INNER_HEIGTH = 1280;
    private final int FULL_INNER_WIDTH = 1920;
    private final int BORDER_THICKNESS = 30;
    private final int CORNER_OFFSET = 11;
    private final int INSET = 4;
    private final int INSET_CORNER_AFTER = 15;

    private final EventPlayerFrame control;

    public EventPlayerFrameSkin(EventPlayerFrame control) {
        super(control);
        this.control = control;
        Group g = new Group();
        g.setManaged(false);
        initializeFrame(g);
        addResizeHandler(g);
        getChildren().add(g);
    }

    private void addResizeHandler(final Group g) {
        control.parentWidthProperty().addListener((observable, oldValue, newValue) -> {
            g.getChildren().clear();
            initializeFrame(g);
        });
    }

    private void initializeFrame(Group g) {
        final double targetWidth = calculateTargetWidth();
        int nbHorizElements = calculateNumberHorizontalElements(targetWidth);
        double innerWidthUnscaled = 2 * CORNER_OFFSET + nbHorizElements * FRAME_BORDER_LENGTH;
        Preconditions.checkArgument(innerWidthUnscaled >= targetWidth, "The unscaled width ("+innerWidthUnscaled+") must be larger than the target width ("+targetWidth+")");
        double scale = targetWidth/innerWidthUnscaled;
        double innerWidthScaled = innerWidthUnscaled * scale;
        double innerHeigthScaled = calculateTargetHeigth(innerWidthScaled);
        int nbVertElements = calculateNumberVerticalElements(innerHeigthScaled);
        Image tl = createImage("DecoratedFrame-Corner-TL.png");
        Image tr = createImage("DecoratedFrame-Corner-TR.png");
        Image bl = createImage("DecoratedFrame-Corner-BL.png");
        Image br = createImage("DecoratedFrame-Corner-BR.png");
        Image top = createImage("DecoratedFrame-Top.png");
        Image left = createImage("DecoratedFrame-Left.png");
        Image right = createImage("DecoratedFrame-Right.png");
        Image bottom = createImage("DecoratedFrame-Bottom.png");
        double offsetX = 0;
        double offsetY = 0;
        // Draw top border
        ImageView imgView = createScaledView(tl, scale);
        double width = tl.getWidth() * scale;
        imgView.setLayoutX(offsetX);
        imgView.setLayoutY(offsetY);
        g.getChildren().addAll(imgView);
        offsetX += width - INSET_CORNER_AFTER * scale;
        for (int i = 0; i < nbHorizElements; i++) {
            imgView = createScaledView(top, scale);
            width = top.getWidth() * scale;
            imgView.setLayoutX(offsetX);
            imgView.setLayoutY(offsetY);
            g.getChildren().addAll(imgView);
            offsetX += width - INSET * scale;
        }
        offsetX -= 2;
        final double offsetRightBorder = offsetX;
        imgView = createScaledView(tr, scale);
        width = tr.getWidth() * scale;
        imgView.setLayoutX(offsetX);
        imgView.setLayoutY(offsetY);
        g.getChildren().addAll(imgView);
        offsetX = 0;
        offsetY += width - INSET_CORNER_AFTER * scale;
        // Left and right
        double height = width;
        for (int i = 0; i < nbVertElements; i++) {
            imgView = createScaledView(left, scale);
            height = left.getHeight() * scale;
            imgView.setLayoutX(0);
            imgView.setLayoutY(offsetY);
            g.getChildren().addAll(imgView);
            imgView = createScaledView(right, scale);
            imgView.setLayoutX(offsetRightBorder);
            imgView.setLayoutY(offsetY);
            g.getChildren().addAll(imgView);
            offsetY += height - INSET * scale;
        }
        // Bottom
        offsetY -= 2;
        imgView = createScaledView(bl, scale);
        width = bl.getWidth() * scale;
        imgView.setLayoutX(offsetX);
        imgView.setLayoutY(offsetY);
        g.getChildren().addAll(imgView);
        offsetX += width - INSET_CORNER_AFTER * scale;
        for (int i = 0; i < nbHorizElements; i++) {
            imgView = createScaledView(bottom, scale);
            width = bottom.getWidth() * scale;
            imgView.setLayoutX(offsetX);
            imgView.setLayoutY(offsetY);
            g.getChildren().addAll(imgView);
            offsetX += width - INSET * scale;
        }
        offsetX -= 2;
        imgView = createScaledView(br, scale);
        width = br.getWidth() * scale;
        imgView.setLayoutX(offsetX);
        imgView.setLayoutY(offsetY);
        g.getChildren().addAll(imgView);
        // Calculate dimensions
        Bounds frameBounds = g.getLayoutBounds();
        double borderInset = scale * BORDER_THICKNESS;
        double backgroundInset = scale * BORDER_THICKNESS/2;
        Dimension2D dim = new Dimension2D(frameBounds.getWidth() - 2 * borderInset, frameBounds.getHeight() - 2 * borderInset);
        ((ObjectProperty)control.innerDimensionProperty()).setValue(dim);
        ((ObjectProperty)control.insetMediaContentProperty()).setValue(new Point2D(borderInset, borderInset));
        ((ObjectProperty)control.insetBackgroundProperty()).setValue(new Point2D(backgroundInset, backgroundInset));
        dim = new Dimension2D(frameBounds.getWidth() - 2 * backgroundInset, frameBounds.getHeight() - 2 * backgroundInset);
        ((ObjectProperty)control.backgroundDimensionProperty()).setValue(dim);
    }

    private ImageView createScaledView(Image img, double scale) {
        ImageView view = new ImageView(img);
        view.setScaleX(scale);
        view.setScaleY(scale);
        return view;
    }

    /**
     * Create the unscaled image
     * @param imgFileName
     * @return
     */
    private Image createImage(String imgFileName) {
        InputStream is = getClass().getResourceAsStream(imgFileName);
        Image unscaled = new Image(is);
        return unscaled;
    }

    private double calculateTargetWidth() {
        double parentWidth = control.getParentWidth();
        return parentWidth * SCALE_FACTOR;
    }

    private int calculateNumberHorizontalElements(double targetWidth) {
        int x = (int) Math.max(Math.ceil((targetWidth - 2 * CORNER_OFFSET)/FRAME_BORDER_LENGTH), 0);
        return x;
    }

    private double calculateTargetHeigth(double targetWidth) {
        double x = targetWidth * FULL_INNER_HEIGTH / FULL_INNER_WIDTH;
        return x;
    }

    private int calculateNumberVerticalElements(double targetHeigth) {
        int x = (int) Math.max(Math.ceil((targetHeigth - 2 * CORNER_OFFSET)/FRAME_BORDER_LENGTH), 0);
        return x;
    }

}
