package ch.sahits.game.javafx.util;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.javafx.control.MoneyTransfer;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.DynamicTextTableCell;
import ch.sahits.game.javafx.model.ITableCell;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import javafx.geometry.HPos;
import javafx.geometry.VPos;
import javafx.scene.Node;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

/**
 * Utility class facilitating some common tasks required by multiple components.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 14, 2013
 */
@Component
@Lazy
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class JavaFXUtils {
    @Autowired
    private ApplicationContext context;
    /**
     * Adding a table cell to the grid
     * @param grid grid pant to which to add.
     * @param rowNum row index
     * @param row table row
     * @param col column index
     * @param cell the node is pased on
     * @param hAlignment horizontal alignment
     * @param textStyleClass
     */
    public void addCellToGridPane(GridPane grid,  int rowNum, TableRow row, int col, ITableCell cell, HPos hAlignment, String textStyleClass) {
        if (cell instanceof StaticTextTableCell) {
            Text text = new Text(((StaticTextTableCell)cell).getValue());
            text.getStyleClass().add(textStyleClass);
            addCellToGridPane(rowNum, row, col, cell, text, grid, hAlignment);
        } else if (cell instanceof DynamicTextTableCell) {
            DynamicTextTableCell dynCell = (DynamicTextTableCell) cell;
            Text text = new Text();
            text.textProperty().bind(dynCell.valueProperty());
            text.getStyleClass().add(textStyleClass);
            addCellToGridPane(rowNum, row, col, cell, text, grid, hAlignment);
        } else if (cell instanceof ControlTableCell) {
            final Node control = ((ControlTableCell) cell).getControl();
            addCellToGridPane(rowNum, row, col, cell, control, grid, hAlignment);
        } else {
            throw new RuntimeException("Not implemented");
        }
    }
    /**
     * Add a node to the grid pane with respect to the different constraints,
     * that might be defined.
     * @param rowNum row index
     * @param row table row
     * @param col column index
     * @param cell the node is pased on
     * @param node node to be added
     * @param grid grid pant to which to add.
     * @param halignment horizontal alignment
     */
    private void addCellToGridPane(int rowNum, TableRow row, int col,
                                   ITableCell cell, Node node, GridPane grid, HPos halignment) {
        if (row != null && row.hasConstraint(cell)) {
            switch (row.getCellConstraint(cell)) {
                case COLSPAN2:
                    grid.add(node, col,rowNum+1, 2, 1);
                    GridPane.setColumnSpan(node, 2);
                    break;
                case COLSPAN4:
                    grid.add(node, col,rowNum+3, 2, 1);
                    GridPane.setColumnSpan(node, 4);
                    break;
                default:
                    grid.add(node, col, rowNum+1);
                    break;
            }
        } else {
            grid.add(node, col, rowNum+1);
        }
        GridPane.setHalignment(node, halignment);
        GridPane.setValignment(node, VPos.CENTER);
    }
    /**
     * Create a grid pane from a supplied table model.
     * @param model base model
     * @return grid pane
     */
    public GridPane createGridPaneFromModel(Table model) {
        TableHeader header = model.getHeader();
        GridPane grid = new GridPane();
        //grid.setGridLinesVisible(true);
        RowConstraints rowConstraint = new RowConstraints(48);
        grid.getRowConstraints().add(rowConstraint);
        rowConstraint = new RowConstraints(24);
        // Header
        for (int i = 0; i < model.getNumberOfColumns(); i++) {
            int colWidth = model.getColumnWidth(i);
            ColumnConstraints colConstraint = new ColumnConstraints(colWidth);
            colConstraint.setHalignment(header.getAligenment(i));
            grid.getColumnConstraints().add(colConstraint);
            ITableCell cell = header.get(i);
            final String textStyleClass = "tableHeader";
            addCellToGridPane(grid, -1, null, i, cell, header.getAligenment(i), textStyleClass);
        }
        // Table
        for (int rowNum = 0; rowNum < model.getNumberOfRows(); rowNum++) {
            TableRow row = model.get(rowNum);
            grid.getRowConstraints().add(rowConstraint);
            final int iterationLimit = Math.min(model.getNumberOfColumns(), row.size());
            for (int col = 0; col < iterationLimit; col++) {

                ITableCell cell = row.get(col);
                final HPos hAlignment = model.getAligenment(col);
                final String textStyleClass = "tableCell";
                addCellToGridPane(grid, rowNum, row, col, cell, hAlignment, textStyleClass);
            }
        }

        grid.setHgap(5);
        return grid;
    }

    /**
     * Create a new instance for the money transfer.
     * @return
     */
    public MoneyTransfer getMoneyTransfer(IDataImageLoader imageLoader) {
        return new MoneyTransfer(imageLoader);
    }
}
