package ch.sahits.game.openpatrician.javafx.service;

import ch.sahits.game.openpatrician.javafx.control.BaseItemNavigatorLabelWrapper;
import ch.sahits.game.openpatrician.javafx.control.IItemNavigatorLabelWrapper;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;

import java.util.ArrayList;
import java.util.List;

/**
 * Factory class to create a label wrapper list from a normal list.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 20, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public class ItemNavigatorLabelWrapperFactory {
    /**
     * Create a list of label wrapper for ships.
     * @param ships
     * @return
     */
    public List<IItemNavigatorLabelWrapper<IShip>> createListForVessels(List<IShip> ships) {
        List<IItemNavigatorLabelWrapper<IShip>> list = new ArrayList<>();
        for (IShip ship : ships) {
            list.add(new ShipNavigatorWrapper(ship));
        }
        return list;
    }

    /**
     * Create a list of label wrapper for citizens, which include {@link ICitizen}s as well as {@link ch.sahits.game.openpatrician.model.IPlayer}s.
     * @param persons
     * @return
     */
    public List<IItemNavigatorLabelWrapper<IPlayer>> createListForPlayer(List<IPlayer> persons) {
        List<IItemNavigatorLabelWrapper<IPlayer>> list = new ArrayList<>();
        for (IPlayer person : persons) {
            list.add(new PersonNavigatorWrapper(person));
        }
        return list;
    }

    /**
     * Create a list of label wrapper for players, which include {@link ICitizen}s as well as {@link ch.sahits.game.openpatrician.model.IPlayer}s.
     * The name of the player is not displayed.
     * @param persons
     * @return
     */
    public List<IItemNavigatorLabelWrapper<IPlayer>> createNonLabeledListForPlayer(List<IPlayer> persons) {
        List<IItemNavigatorLabelWrapper<IPlayer>> list = new ArrayList<>();
        for (IPlayer person : persons) {
            list.add(new NonDisplayablePersonNavigatorWrapper(person));
        }
        return list;
    }

    private static class ShipNavigatorWrapper extends BaseItemNavigatorLabelWrapper<IShip> {
        public ShipNavigatorWrapper(IShip item) {
            super(item);
        }

        @Override
        public String getLabel() {
            return getItem().getName();
        }
    }

    private static class VesselNavigatorWrapper extends BaseItemNavigatorLabelWrapper<INavigableVessel> {
        public VesselNavigatorWrapper(INavigableVessel item) {
            super(item);
        }

        @Override
        public String getLabel() {
            return getItem().getName();
        }
    }

    private static class PersonNavigatorWrapper extends BaseItemNavigatorLabelWrapper<IPlayer> {
        public PersonNavigatorWrapper(IPlayer item) {
            super(item);
        }

        @Override
        public String getLabel() {
            return getItem().getName() + " " + getItem().getLastName();
        }
    }

    private static class NonDisplayablePersonNavigatorWrapper extends BaseItemNavigatorLabelWrapper<IPlayer> {
        public NonDisplayablePersonNavigatorWrapper(IPlayer item) {
            super(item);
        }

        @Override
        public String getLabel() {
            return "";
        }
    }
}
