package ch.sahits.game.openpatrician.javafx.service;

import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.model.EDialogType;
import ch.sahits.game.openpatrician.javafx.model.ENoticeBoardType;
import ch.sahits.game.openpatrician.javafx.model.NoticeBoardMenuEntry;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.CitiesState;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.IBuyer;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.IInformant;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IPirate;
import ch.sahits.game.openpatrician.model.people.IPirateHunter;
import ch.sahits.game.openpatrician.model.people.ISailors;
import ch.sahits.game.openpatrician.model.people.ISideRoomPerson;
import ch.sahits.game.openpatrician.model.people.ITrader;
import ch.sahits.game.openpatrician.model.people.ITraveler;
import ch.sahits.game.openpatrician.model.people.ITreasureMapOwner;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import ch.sahits.game.openpatrician.model.people.impl.CaptainsState;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.MessageSource;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;

/**
 * Provide the menu items fo the tavern notice board menu.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 19, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class NoticeBoardTavernMenuProvider implements INoticeBoardMenuProvider {
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private CaptainsState captainsState;
    @Autowired
    private CitiesState cityState;
    @Override
    public LinkedHashSet<NoticeBoardMenuEntry> createMenuEntries(ICityPlayerProxyJFX proxy, ENoticeBoardType type) {
        LinkedHashSet<NoticeBoardMenuEntry> entries = new LinkedHashSet<>();

        final TavernState tavernState = getTavernState(proxy.getCity());
        Optional<ICaptain> captain = captainsState.getCaptain(proxy.getCity());
        List<IPerson> present = tavernState.presentPersonsProperty();

        List<ISideRoomPerson> sideRoom = Lists.newArrayList();
        List<IPerson> mainRoom = Lists.newArrayList();
        IPlayer player = proxy.getPlayer();
        collectPresentPersons(tavernState, present, sideRoom, mainRoom, player);
        if (captain.isPresent()) {
            mainRoom.add(captain.get());
        }
        for (IPerson pers : mainRoom) {
            String s = messageSource.getMessage(pers.getClass().getName() + ".noticeboardTitle", new Object[0], locale.getCurrentLocal());
            NoticeBoardMenuEntry entry = createNoticeBoardEntryMainRoom(s, pers, tavernState, player);
            entries.add(entry);
        }
        if (!sideRoom.isEmpty()) {
            String s = messageSource.getMessage("ch.sahits.game.openpatrician.display.notice.NoticeBoardFactory.sideRoom", new Object[]{}, locale.getCurrentLocal());
            NoticeBoardMenuEntry entry = createNoticeBoardEntrySideRoom(s, sideRoom.get(0), tavernState, player);
            entries.add(entry);
        }

        return entries;
    }
    @VisibleForTesting
    void collectPresentPersons(TavernState tavernState, List<IPerson> present, List<ISideRoomPerson> sideRoom, List<IPerson> mainRoom, IPlayer player) {
        for (IPerson pers : present) {
            if (pers instanceof ISideRoomPerson) {
                if (!tavernState.isTalkingToOtherPlayer((ISideRoomPerson) pers, player)) {
                    if ((pers instanceof ITreasureMapOwner || pers instanceof IPirateHunter) && player.getSegmentedMap() != null) {
                        continue; // only add Treasure map owner or pirate hunter if the player owns no map piece.
                    } else {
                        sideRoom.add((ISideRoomPerson) pers);
                    }
                }
            } else {
                mainRoom.add(pers);
            }
        }
    }

    private TavernState getTavernState(ICity city) {
        for (CityState state : cityState.getCityEngineStates()) {
            if (state.getCity().equals(city)) {
                return state.getTavernState();
            }
        }
        return null;
    }
    private NoticeBoardMenuEntry createNoticeBoardEntryMainRoom(String s, IPerson showingPerson,
                                                                  TavernState tavernState,
                                                                  IPlayer currentPlayer) {
        DecoratedText label = textFactory.createSimpleDecoratedText(s);
        EDialogType dialogType = getDialogTypeMainRoom(showingPerson);
        NoticeBoardMenuEntry entry = new NoticeBoardMenuEntry(dialogType);
        entry.setLabel(label);
        Object[] args = new Object[]{showingPerson, tavernState, currentPlayer, dialogType};
        final Runnable runnable1 = (Runnable) context.getBean("tavernDialogChangeAction", args);
        EventHandler<MouseEvent> openDialogEvenHandler = evt -> runnable1.run();
        entry.setOnMouseSelect(openDialogEvenHandler);
        return entry;
    }
    private NoticeBoardMenuEntry createNoticeBoardEntrySideRoom(String s, IPerson showingPerson,
                                                                TavernState tavernState, IPlayer player) {
        DecoratedText label = textFactory.createSimpleDecoratedText(s);
        NoticeBoardMenuEntry entry = new NoticeBoardMenuEntry(EDialogType.TAVERN_SIDE_ROOM);
        entry.setLabel(label);
        Object[] args = new Object[]{showingPerson, tavernState, player};
        final Runnable runnable1 = (Runnable) context.getBean("tavernDialogTalkToSideroomPersonAction", args);
        EventHandler<MouseEvent> openDialogEvenHandler = evt -> runnable1.run();
        entry.setOnMouseSelect(openDialogEvenHandler);
        return entry;
    }
    private EDialogType getDialogTypeMainRoom(IPerson showingPerson) {
        EDialogType selected = null;
        if (showingPerson instanceof IBuyer) {
            selected = EDialogType.TAVERN_BUYER;
        }
        if (showingPerson instanceof ISailors) {
            selected = EDialogType.TAVERN_SAILORS;
        }
        if (showingPerson instanceof IWeaponsDealer) {
            selected = EDialogType.TAVERN_WEAPONS_DEALER;
        }
        if (showingPerson instanceof IInformant) {
            selected = EDialogType.TAVERN_INFORMANT_1;
        }
        if (showingPerson instanceof ICaptain) {
            selected = EDialogType.TAVERN_CAPTAIN;
        }
        if (showingPerson instanceof IPirate) {
            selected = EDialogType.TAVERN_PIRATE;
        }
        if (showingPerson instanceof ITraveler) {
            selected = EDialogType.TAVERN_TRAVELER;
        }
        if (showingPerson instanceof ITrader) {
            selected = EDialogType.TAVERN_TRADER;
        }
        return selected;
    }
}
