package ch.sahits.game.openpatrician.javafx.control;

import ch.sahits.game.openpatrician.javafx.control.skin.DisplayMessageOverlayLabelSkin;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.event.EventHandler;
import javafx.scene.control.Control;
import javafx.scene.control.Skin;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Font;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * This control represents a display message on a wax seal. The size of the seal is calculated based on the size of the
 * text. The label may be a link indicated by the appropriate property. If so the onMouseRelease event handler is used.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Mar 08, 2015
 */
@ClassCategory(EClassCategory.JAVAFX)
public class DisplayMessageOverlayLabel extends Control {
    private StringProperty text = new SimpleStringProperty(this, "text", "");
    private BooleanProperty linkText = new SimpleBooleanProperty(this, "link-text", false);
    @ObjectPropertyType(EventHandler.class)
    private ObjectProperty<EventHandler<MouseEvent>> onMouseRelease = new SimpleObjectProperty<>(this, "on-mouse-released");
    private final TextSizingUtility sizeing;
    public DisplayMessageOverlayLabel(String text, TextSizingUtility sizeing) {
        this.text.setValue(text);
        this.sizeing = sizeing;
        getStylesheets().add(this.getClass().getResource("DisplayMessageOverlayLabel.css").toExternalForm());
        getStyleClass().addAll("display-message");
    }

    public String getText() {
        return text.get();
    }

    public StringProperty textProperty() {
        return text;
    }

    public void setText(String text) {
        this.text.set(text);
    }

    public boolean isLinkText() {
        return linkText.get();
    }

    public BooleanProperty linkTextProperty() {
        return linkText;
    }

    public void setLinkText(boolean linkText) {
        this.linkText.set(linkText);
    }


    public void setOnMouseRelease(EventHandler<MouseEvent> onMouseRelease) {
        this.onMouseRelease.set(onMouseRelease);
    }

    @Override
    protected Skin<?> createDefaultSkin() {
        return new DisplayMessageOverlayLabelSkin(this, sizeing);
    }

    private StyleableObjectProperty<Font> font;

    public Font getFont() {
        return font == null ? Font.getDefault() : font.get();
    }
    public void setFont(Font font) {
        this.font.set(font);
    }
    public StyleableObjectProperty<Font> fontProperty() {
        if (font == null) {
            font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, DisplayMessageOverlayLabel.this, "font", Font.getDefault());
        }
        return font;
    }

    private static class StyleableProperties {
        private static final CssMetaData< DisplayMessageOverlayLabel, Font> FONT =
                new CssMetaData< DisplayMessageOverlayLabel, Font>("-fx-font",
                        StyleConverter.getFontConverter(), Font.getDefault()) {
                    @Override
                    public boolean isSettable(DisplayMessageOverlayLabel control) {
                        return control.font == null || !control.font.isBound();
                    }
                    @Override
                    public StyleableProperty<Font> getStyleableProperty(DisplayMessageOverlayLabel control) {
                        return control.fontProperty();
                    }
                };
        private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;
        static {
            final List<CssMetaData<? extends Styleable, ?>> styleables =
                    new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
            Collections.addAll(styleables,
                    FONT
            );
            STYLEABLES = Collections.unmodifiableList(styleables);
        }
    }
    @Override
    public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
        return getClassCssMetaData();
    }
    public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
        return StyleableProperties.STYLEABLES;
    }
}
