package ch.sahits.game.openpatrician.javafx.control;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.scene.layout.BorderPane;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.ArrayList;
import java.util.List;

/**
 * Control that allows the circular navigation of a list.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 20, 2016
 */
@ClassCategory(EClassCategory.JAVAFX)
public class ItemNavigator<T> extends BorderPane {
    private final Logger logger = LogManager.getLogger(getClass());

    private final ObservableList<IItemNavigatorLabelWrapper<T>> list = FXCollections.observableList(new ArrayList<>());
    private IntegerProperty currentPosition = new SimpleIntegerProperty(0);

    public ItemNavigator(List<IItemNavigatorLabelWrapper<T>> list) {
        this.list.setAll(list);
        initializeControl();
    }

    public int getCurrentPosition() {
        return currentPosition.get();
    }

    public IntegerProperty currentPositionProperty() {
        return currentPosition;
    }

    private void next() {
        try {
            if (currentPosition.get() == list.size() - 1) {
                currentPosition.setValue(0);
            } else {
                currentPosition.setValue(currentPosition.get() + 1);
            }
        } catch (RuntimeException e) {
            logger.error("Failed to switch to next item", e);
        }
    }

    private void previous() {
        try {
            if (currentPosition.get() == 0) {
                currentPosition.setValue(list.size() - 1);
            } else {
                currentPosition.setValue(currentPosition.get() - 1);
            }
        } catch (RuntimeException e) {
            logger.error("Failed to switch to previous item", e);
        }
    }

    private void initializeControl() {
        if (!list.isEmpty()) {
            final Text label = new Text(list.get(getCurrentPosition()).getLabel());
            StringBinding labelBinding = new StringBinding() {
                {
                    super.bind(currentPosition);
                }

                @Override
                protected String computeValue() {
                    return list.get(currentPosition.intValue()).getLabel();
                }
            };
            label.textProperty().bind(labelBinding);
            label.getStyleClass().add("dialogText");
            label.setId("navigation-label");
            setCenter(label);

            BooleanBinding visibleNavigation = new BooleanBinding() {
                {
                    super.bind(list);
                }

                @Override
                protected boolean computeValue() {
                    return !list.isEmpty();
                }
            };

            final OpenPatricianSmallWaxButton previous = new OpenPatricianSmallWaxButton("<");
            previous.setId("previous");
            previous.getStyleClass().add("actionButton");
            previous.setOnAction(event -> previous());
            previous.visibleProperty().bind(visibleNavigation);
            setLeft(previous);
            final OpenPatricianSmallWaxButton next = new OpenPatricianSmallWaxButton(">");
            next.setId("next");
            next.getStyleClass().add("actionButton");
            next.setOnAction(event -> next());
            next.visibleProperty().bind(visibleNavigation);
            setRight(next);
        }
    }

    public T getCurrent() {
        return list.get(currentPosition.intValue()).getItem();
    }
}
