package ch.sahits.game.openpatrician.javafx.control;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.javafx.control.skin.ShipIconSkin;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.scene.control.Control;
import javafx.scene.control.Skin;

import java.net.URL;

/**
 * The ShipIcon represents the icon of the traveling vessel with all attached information. The origin of this control is
 * the center of the ship icon itself. This control also includes a shield indicating the owner, convoy size and pirate
 * state.
 * The <code>vessel</code>'s UUID will be assigned as id to this control.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Nov 18, 2017
 */
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class ShipIcon extends Control {

    private final INavigableVessel vessel;
    private final IHumanPlayer player;
    private final IDataImageLoader imageLoader;

    private ShipIconSkin skin;

    private BooleanProperty travelingEast = new SimpleBooleanProperty(false);
    private DoubleProperty scale = new SimpleDoubleProperty(1);

    public ShipIcon(INavigableVessel vessel, IHumanPlayer player, IDataImageLoader imageLoader) {
        this.vessel = vessel;
        this.imageLoader = imageLoader;
        this.player = player;
        setId(vessel.getUuid());
        createDefaultSkin();
    }

    /**
     * Adjust the layout position of this control to match the vessels position.
     */
    public void updatePosition() {
        skin.updateShipPosition();
    }

    public void setTravelingEast(boolean travelingEast) {
        this.travelingEast.set(travelingEast);
    }

    public boolean isTravelingEast() {
        return travelingEast.get();
    }

    public BooleanProperty travelingEastProperty() {
        return travelingEast;
    }

    public double getScale() {
        return scale.get();
    }

    public DoubleProperty scaleProperty() {
        return scale;
    }

    public void setScale(double scale) {
        this.scale.set(scale);
    }

    public INavigableVessel getVessel() {
        return vessel;
    }

    public IHumanPlayer getPlayer() {
        return player;
    }

    @Override
    public String getUserAgentStylesheet() {
        URL cssURL = getClass().getResource(getClass().getSimpleName()+".css");
        return cssURL.toExternalForm();
    }

    @Override
    public Skin<?> createDefaultSkin() {
        if (skin == null) {
            skin = new ShipIconSkin(this, imageLoader);
        }
        return skin;
    }
}
