package ch.sahits.game.openpatrician.javafx.control;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.city.impl.ECityState;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.javafx.IJavaFXApplicationThreadExecution;
import ch.sahits.game.openpatrician.utilities.javafx.JavaFXApplicationThreadExecution;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.property.ObjectProperty;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.VBox;

import java.util.Iterator;

/**
 * Icons of the city state.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Jan 27, 2018
 */
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class CityIcons extends VBox {

    public static final String MISSING_WARE_ID = "missingWare";
    private static final String CITY_EVENT_ID = "cityEvent";
    private final CityState cityState;
    private IJavaFXApplicationThreadExecution threadExecution;

    private final IDataImageLoader imageLoader;

    public CityIcons(CityState cityState, IDataImageLoader imageLoader) {
        super(2);
        this.cityState = cityState;
        this.imageLoader = imageLoader;
        threadExecution = new JavaFXApplicationThreadExecution();
        updateMissingWare();
        updateCityEvent();
        cityState.longestMissingWaresBinding().addListener((observable, oldValue, newValue) -> updateMissingWare());
        cityState.cityEventProperty().addListener((observable, oldValue, newValue) -> updateCityEvent());
    }

    private void updateCityEvent() {
        threadExecution.execute(() -> {
            ObjectProperty<ECityState> cityEvent = cityState.cityEventProperty();
            for (Iterator<Node> iterator = getChildren().iterator(); iterator.hasNext(); ) {
                Node child = iterator.next();
                if (child.getId().equals(CITY_EVENT_ID)) {
                    iterator.remove();
                    break;
                }
            }
            if (cityEvent.get() != null) {
                // display event icon
                Image iconImg = imageLoader.getImage("icons/32/" + getEventIconName(cityEvent.get()));
                ImageView imgView = new ImageView(iconImg);
                getChildren().add(imgView);
            }
        });
    }

    private void updateMissingWare() {
        threadExecution.execute(() -> {
            ObjectBinding<IWare> missingWareeBinding = cityState.longestMissingWaresBinding();
            for (Iterator<Node> iterator = getChildren().iterator(); iterator.hasNext(); ) {
                Node child = iterator.next();
                if (child.getId().equals(MISSING_WARE_ID)) {
                    iterator.remove();
                    break;
                }
            }
            if (missingWareeBinding.get() != null) {
                if (cityState.famineBinding().get()) {
                    // display famine icon
                    Image famine = imageLoader.getImage("icons/32/icon_famine");
                    ImageView famineView = new ImageView(famine);
                    famineView.setId(MISSING_WARE_ID);
                    getChildren().add(0, famineView);
                } else {
                    // display missing ware icon
                    String wareIconName = "icons/32/" + getIconName(missingWareeBinding.get());
                    Image missingWare = imageLoader.getImage(wareIconName);
                    ImageView missingWareView = new ImageView(missingWare);
                    missingWareView.setId(MISSING_WARE_ID);
                    getChildren().add(0, missingWareView);
                }
            }
        });
    }


    private String getIconName(IWare ware) {
        switch ((EWare)ware) {
            case BEER:
                return "icon_beer";
            case BRICK:
                return "icon_bricks";
            case CLOTH:
                return "icon_cloth";
            case FISH:
                return "icon_fish";
            case FISH_OIL:
                return "icon_fish_oil";
            case FUR:
                return "icon_fur";
            case GRAIN:
                return "icon_grain";
            case HEMP:
                return "icon_hemp";
            case HONEY:
                return "icon_honey";
            case IRON:
                return "icon_iron";
            case IRONORE:
                return "icon_iron_ore";
            case LEATHER:
                return "icon_leather";
            case MEAT:
                return "icon_meat";
            case PITCH:
                return "icon_pitch";
            case POTTERY:
                return "icon_pottery";
            case SALT:
                return "icon_salt";
            case SPICE:
                return "icon_spices";
            case WINE:
                return "icon_wine";
            case WOOD:
                return "icon_wood";
            case WOOL:
                return "icon_wool";
            default:
                throw new IllegalArgumentException("Unhandled ware: "+ware);
        }
    }

    private String getEventIconName(ECityState state) {
        switch (state) {
            case SIEGE:
                return "icon_siege";
            case BLOCKADE:
                return "icon_blockade";
            case FROZEN_HARBOR:
                return "icon_frozen";
            case PLAGUE:
                return "icon_plague";
            case FIRE:
                return "icon_fire";
            default:
                throw new IllegalArgumentException("Unhandled state: "+state);
        }
    }
}
