package ch.sahits.game.openpatrician.javafx.control;

import ch.sahits.game.openpatrician.javafx.control.skin.EventPlayerFrameSkin;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.ReadOnlyDoubleProperty;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import javafx.scene.control.Control;
import javafx.scene.control.Skin;

/**
 * Provide the frame control for the media event. The size of the frame is dependent on
 * the parent size. It should be in the width of 50% - 66% of the parent.
 * The frame is supposed to work together in a stacked layout with a background at position
 * {@link #getInsetBackground()} and dimension {@link #getBackgroundDimension()} and the media
 * content at {@link #getInsetMediaContent()} with a maximum space of {@link #getInnerDimension()}.
 * This control will then placed on top partially covering the background and media content.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 *         Created on Jan 03, 2017
 */
@ClassCategory(EClassCategory.JAVAFX)
public class EventPlayerFrame extends Control {

    private final DoubleProperty parentWidth = new SimpleDoubleProperty();
    private final ObjectProperty<Dimension2D> innerDimension = new SimpleObjectProperty<>(new Dimension2D(0,0));
    private final ObjectProperty<Dimension2D> backgroundDimension = new SimpleObjectProperty<>(new Dimension2D(0,0));
    private final ObjectProperty<Point2D> insetMediaContent = new SimpleObjectProperty<>(new Point2D(0,0));
    private final ObjectProperty<Point2D> insetBackground = new SimpleObjectProperty<>(new Point2D(0, 0));


    public EventPlayerFrame(ReadOnlyDoubleProperty parentWidth) {
        this.parentWidth.bind(parentWidth);
    }

    @Override
    protected Skin<?> createDefaultSkin() {
        return new EventPlayerFrameSkin(this);
    }

    public double getParentWidth() {
        return parentWidth.get();
    }

    public DoubleProperty parentWidthProperty() {
        return parentWidth;
    }

    /**
     * Retrieve the dimension of the inner border of the frame.
     * @return
     */
    public Dimension2D getInnerDimension() {
        return innerDimension.get();
    }

    /**
     * Property of the inner dimensions.
     * @return
     */
    public ReadOnlyObjectProperty<Dimension2D> innerDimensionProperty() {
        return innerDimension;
    }

    /**
     * Retrieve the inset coordinates for the media content.
     * @return
     */
    public Point2D getInsetMediaContent() {
        return insetMediaContent.get();
    }

    /**
     * Retrieve the property for the coordinates of the media content.
     * @return
     */
    public ReadOnlyObjectProperty<Point2D> insetMediaContentProperty() {
        return insetMediaContent;
    }

    /**
     * Retrieve the inset coordinates for the background.
     * @return
     */
    public Point2D getInsetBackground() {
        return insetBackground.get();
    }

    /**
     * Retrieve the property for the inset coordinates for the background.
     * @return
     */
    public ReadOnlyObjectProperty<Point2D> insetBackgroundProperty() {
        return insetBackground;
    }

    /**
     * Retrieve the dimensions of the background.
     * @return
     */
    public Dimension2D getBackgroundDimension() {
        return backgroundDimension.get();
    }

    /**
     * Retrieve the property for the background dimension.
     * @return
     */
    public ReadOnlyObjectProperty<Dimension2D> backgroundDimensionProperty() {
        return backgroundDimension;
    }
}
