package ch.sahits.game.openpatrician.javafx.control;

import ch.sahits.game.openpatrician.javafx.control.skin.OpenPatricianSliderSkin;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.annotation.ObjectPropertyType;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleListProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.css.CssMetaData;
import javafx.css.SimpleStyleableObjectProperty;
import javafx.css.StyleConverter;
import javafx.css.Styleable;
import javafx.css.StyleableObjectProperty;
import javafx.css.StyleableProperty;
import javafx.scene.control.Control;
import javafx.scene.control.Skin;
import javafx.scene.text.Font;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@ClassCategory(EClassCategory.JAVAFX)
public class OpenPatricianSlider extends Control {
	@ListType(String.class)
	private final ObservableList<String> values;

	private final IntegerProperty selectedIndex;
	private final ReadOnlyIntegerProperty initialWidth;
	@ObjectPropertyType(Font.class)
	private StyleableObjectProperty<Font> font;


	public OpenPatricianSlider(int width) {
		values = new SimpleListProperty<>(this, "options", FXCollections.observableList(new ArrayList<String>()));
		initialWidth = new SimpleIntegerProperty(this,"initialWidth", width);
		selectedIndex = new SimpleIntegerProperty(this, "selectedIndex", -1);
		
		this.getStyleClass().add(this.getClass().getSimpleName());
	}

	@Override
	public String getUserAgentStylesheet() {
        URL cssURL = getClass().getResource("/ch/sahits/game/openpatrician/javafx/control/"+getClass().getSimpleName()+".css");
		return cssURL.toExternalForm();
	}

	@Override
	protected Skin<?> createDefaultSkin() {
		return new OpenPatricianSliderSkin(this);
	}

	public ObservableList<String> valuesProperty() {
		return values;
	}
	public void setValues(List<String> options) {
		this.values.setAll(options);
		this.selectedIndex.set(0);
	}
	public void setValues(String... options) {
		this.values.setAll(options);
		this.selectedIndex.set(0);
	}
	
	public String getSelectedValue() {
		return values.get(getSelectedIndex());
	}
	
	public IntegerProperty selectedIndexProperty() {
		return selectedIndex;
	}
	public int getSelectedIndex() {
		return selectedIndex.get();
	}
	
	public int getInitialWidth() {
		return initialWidth.get();
	}
	public Font getFont() {
		return font == null ? Font.getDefault() : font.get();
	}

	public void setFont(Font font) {
		this.font.set(font);
	}

	public StyleableObjectProperty<Font> fontProperty() {
		if (font == null) {
			font = new SimpleStyleableObjectProperty<Font>(StyleableProperties.FONT, OpenPatricianSlider.this, "font", Font.getDefault());
		}
		return font;
	}

	private static class StyleableProperties {
		private static final CssMetaData<OpenPatricianSlider, Font> FONT =
				new CssMetaData<OpenPatricianSlider, Font>("-fx-font",
						StyleConverter.getFontConverter(), Font.getDefault()) {
					@Override
					public boolean isSettable(OpenPatricianSlider control) {
						return control.font == null || !control.font.isBound();
					}

					@Override
					public StyleableProperty<Font> getStyleableProperty(OpenPatricianSlider control) {
						return control.fontProperty();
					}
				};
		private static final List<CssMetaData<? extends Styleable, ?>> STYLEABLES;

		static {
			final List<CssMetaData<? extends Styleable, ?>> styleables =
					new ArrayList<CssMetaData<? extends Styleable, ?>>(Control.getClassCssMetaData());
			Collections.addAll(styleables,
					FONT
			);
			STYLEABLES = Collections.unmodifiableList(styleables);
		}
	}

	public List<CssMetaData<? extends Styleable, ?>> getControlCssMetaData() {
		return getClassCssMetaData();
	}

	public static List<CssMetaData<? extends Styleable, ?>> getClassCssMetaData() {
		return StyleableProperties.STYLEABLES;
	}
}
