package ch.sahits.game.openpatrician.javafx.control.skin;

import ch.sahits.game.openpatrician.javafx.control.EventMediaPlayer;
import ch.sahits.game.openpatrician.javafx.control.EventPlayerFrame;
import ch.sahits.game.openpatrician.javafx.control.TextSizingUtility;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.property.ReadOnlyDoubleProperty;
import javafx.geometry.Dimension2D;
import javafx.geometry.Point2D;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.control.SkinBase;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.StackPane;
import javafx.scene.media.Media;
import javafx.scene.media.MediaPlayer;
import javafx.scene.media.MediaView;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import javafx.scene.text.TextAlignment;

import java.io.InputStream;

/**
 * Skin class for the Event media player.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 *         Created on Jan 04, 2017
 */
@ClassCategory(EClassCategory.JAVAFX)
public class EventMediaPlayerSkin extends SkinBase<EventMediaPlayer> {

    private final EventMediaPlayer control;

    private TextSizingUtility textSize = new TextSizingUtility();

    public EventMediaPlayerSkin(EventMediaPlayer control, ReadOnlyDoubleProperty parentWidth) {
        super(control);
        this.control = control;
        Group g = new Group();
        g.setManaged(false);
        initializeControl(g, parentWidth);
        getChildren().add(g);
    }


    private void initializeControl(Group g, ReadOnlyDoubleProperty parentWidth) {
        // The frame
        EventPlayerFrame frame = new EventPlayerFrame(parentWidth);

        // The Background
        Dimension2D backgroundDim = frame.getBackgroundDimension();
        Point2D backgroundInset = frame.getInsetBackground();
        final Rectangle background = new Rectangle(backgroundDim.getWidth(), backgroundDim.getHeight(), Color.SADDLEBROWN);
        background.setLayoutX(backgroundInset.getX());
        background.setLayoutY(backgroundInset.getY());

        // The Title with background
        Label title = new Label(control.getTitle());
        title.getStyleClass().add("title");
        title.textProperty().bind(control.titleProperty());
        Image titleImg = createImage("ParchmentTitleScroll.png");
        final double titleWidth = titleImg.getWidth();
        ImageView titleBackground = new ImageView(titleImg);
        titleBackground.setLayoutX((frame.getWidth() - titleWidth)/2);
        titleBackground.setLayoutY(35);
        title.setLayoutY(47);

        // The description with background
        double scale = frame.getInnerDimension().getWidth()/1920;
        Image descImg = createImage("parchment-background-1920x200.png");
        ImageView descBackground = new ImageView(descImg);
        descBackground.setFitWidth(frame.getInnerDimension().getWidth());
        double yPos = frame.getInsetMediaContent().getY() + frame.getInnerDimension().getHeight() - (200 * scale) + 1;

        Text desc = new Text(control.getDescription());
        desc.getStyleClass().add("description");
        desc.setTextAlignment(TextAlignment.CENTER);
        desc.textProperty().bind(control.descriptionProperty());
        desc.setWrappingWidth(frame.getInnerDimension().getWidth());

        StackPane stack = new StackPane(descBackground, desc);
        stack.setLayoutX(backgroundInset.getX());
        stack.setLayoutY(yPos);

        String resourcePath = control.getType().getResource();
        String sourcePath = getClass().getClassLoader().getResource(resourcePath).toExternalForm();
        Media media = new Media(sourcePath);
        MediaPlayer player = new MediaPlayer(media);
        MediaView mediaView = new MediaView(player);
        mediaView.setPreserveRatio(true);
        mediaView.setFitWidth(frame.getInnerDimension().getWidth());
        mediaView.setLayoutX(frame.getInsetMediaContent().getX());
        mediaView.setLayoutY(frame.getInsetMediaContent().getY());
        player.setAutoPlay(true);

        g.getChildren().addAll(background, stack, mediaView, titleBackground, title, frame);


        final Font openPatrician24 = Font.font("OpenPatrician", 24);
        final Font openPatrician18 = Font.font("OpenPatrician", 18);
        frame.backgroundDimensionProperty().addListener((observable, oldValue, newValue) -> {
            background.setWidth(newValue.getWidth());
            background.setHeight(newValue.getHeight());
            Dimension2D titleDim = textSize.calculate(control.getTitle(), openPatrician24);
            title.setLayoutX((newValue.getWidth() - titleDim.getWidth())/2 + frame.getInsetBackground().getX());
            titleBackground.setLayoutX((newValue.getWidth() - titleWidth)/2 + frame.getInsetBackground().getX() + 20);


        });
        frame.insetBackgroundProperty().addListener((observable, oldValue, newValue) -> {
            background.setLayoutX(newValue.getX());
            background.setLayoutY(newValue.getX());
            Dimension2D titleDim = textSize.calculate(control.getTitle(), openPatrician24);

            title.setLayoutX((frame.getBackgroundDimension().getWidth() - titleDim.getWidth())/2 + newValue.getX());
            titleBackground.setLayoutX((frame.getBackgroundDimension().getWidth() - titleWidth)/2 + newValue.getX() + 20);
        });
        frame.innerDimensionProperty().addListener((observable, oldValue, newValue) -> {
            double localScale = newValue.getWidth()/1920;
            double localYPos = frame.getInsetMediaContent().getY() + 1080 * localScale;
            stack.setLayoutY(localYPos);
            descBackground.setFitWidth(newValue.getWidth() + 4);
            descBackground.setFitHeight(newValue.getHeight() - 1080 * localScale + 2);

            Dimension2D descDimension = textSize.calculate(control.getDescription(), openPatrician18);
            double wrappingWidth = Math.min(descDimension.getWidth(), newValue.getWidth());
            desc.setWrappingWidth(wrappingWidth);
            mediaView.setFitWidth(newValue.getWidth() + 2);
            player.stop();
            player.play();
        });
        frame.insetMediaContentProperty().addListener((observable, oldValue, newValue) -> {
            stack.setLayoutX(frame.getInsetMediaContent().getX());
            double localScale = frame.getInnerDimension().getWidth()/1920;
            double localYPos = newValue.getY() + 1080 * localScale;
            descBackground.setFitHeight(frame.getInnerDimension().getHeight() - 1080 * localScale + 2);
            stack.setLayoutY(localYPos);
            mediaView.setLayoutX(newValue.getX());
            mediaView.setLayoutY(newValue.getY());
            player.stop();
            player.play();
        });
        title.textProperty().addListener((observable, oldValue, newValue) -> {
            Dimension2D titleDim = textSize.calculate(control.getTitle(), openPatrician24);
            title.setLayoutX((frame.getBackgroundDimension().getWidth() - titleDim.getWidth())/2 + frame.getInsetBackground().getX());
        });
        desc.textProperty().addListener((observable, oldValue, newValue) -> {
            Dimension2D descDimension = textSize.calculate(control.getDescription(), openPatrician18);
            double wrappingWidth = Math.min(descDimension.getWidth(), frame.getInnerDimension().getWidth());
            desc.setWrappingWidth(wrappingWidth);
        });
        control.stoppedProperty().addListener((observable, oldValue, newValue) -> {
            if (newValue) {
                player.stop();
            }
        });

    }

    /**
     * Create the unscaled image
     * @param imgFileName
     * @return
     */
    private Image createImage(String imgFileName) {
        InputStream is = getClass().getResourceAsStream(imgFileName);
        Image unscaled = new Image(is);
        return unscaled;
    }
}
