package ch.sahits.game.openpatrician.javafx.control.skin;

import ch.sahits.game.openpatrician.javafx.control.ImageFactory;
import ch.sahits.game.openpatrician.javafx.control.OpenPatricianRadioButton;
import ch.sahits.game.openpatrician.javafx.control.TextSizingUtility;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.geometry.Dimension2D;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.control.SkinBase;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;

import java.io.InputStream;

/**
 * Skin for the OpenPatrician radio button.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Sep 23, 2013
 *
 */
@ClassCategory(EClassCategory.JAVAFX)
public class OpenPatricianRadioButtonSkin extends
		SkinBase<OpenPatricianRadioButton> {
	
	private TextSizingUtility sizing = new TextSizingUtility();
	private ImageFactory imageFactory = new ImageFactory();
    private final ImageView slabView;
    private final Label label;

    public OpenPatricianRadioButtonSkin(final OpenPatricianRadioButton radioButton) {
		super(radioButton);
		String text = radioButton.getText();
		InputStream is = getClass().getResourceAsStream("radioboxbg.png");
        Image rbBG = new Image(is);
        ImageView radioButtonView = new ImageView(rbBG);
        slabView = new ImageView();
        label = new Label(text);


        calculateNewDimenstions(radioButton, text, rbBG);

		radioButton.fontProperty().addListener((observable, oldValue, newValue) -> {
            calculateNewDimenstions(radioButton, text, rbBG);
		});

		label.getStyleClass().add("openPatricianRadioButton");


		final Label selectedLabel;
		if (radioButton.isSelected()) {
			selectedLabel = new Label(radioButton.getSelectedLabel());
		} else {
			selectedLabel = new Label(radioButton.getUnselectedLabel());
		}
		selectedLabel.getStyleClass().add("openPatricianRadioButton-selection");
		selectedLabel.translateXProperty().set(20);
		selectedLabel.translateYProperty().set(32);
		
		radioButton.selectedProperty().addListener(new ChangeListener<Boolean>() {

			@Override
			public void changed(ObservableValue<? extends Boolean> observable,
					Boolean oldValue, Boolean newValue) {
				if (newValue) {
					selectedLabel.setText(radioButton.getSelectedLabel());
				} else {
					selectedLabel.setText(radioButton.getUnselectedLabel());
				}
			}
		});
		
		Group background = new Group(radioButtonView, slabView, label, selectedLabel);
		
		background.setOnMouseReleased(new EventHandler<MouseEvent>() {
			
			@Override
			public void handle(MouseEvent arg0) {
				boolean oldValue = radioButton.selectedProperty().get();
				if (!oldValue) {
					radioButton.setSelected(!oldValue);
				}
			}
			
		});

		background.setManaged(false);
		
		getChildren().add(background);
		

	}

    private void calculateNewDimenstions(OpenPatricianRadioButton radioButton, String text, Image rbBG) {
        InputStream is;
        final int remainWidth = 28; // width that is left from the background
        final int destHight = 27;
        final double xStart = rbBG.getWidth();
        final int yStart = 0;

        Dimension2D dim = sizing.calculate(text, radioButton.getFont());
        double width = dim.getWidth();
        double destWidth = width-remainWidth + 10;

        String fileName = imageFactory.getFileNameExt(destWidth,destHight,remainWidth);
        is = getClass().getResourceAsStream(fileName);
        Image slab = new Image(is, destWidth, destHight, false, true);
        slabView.setImage(slab);
        slabView.translateXProperty().set(xStart);
        slabView.translateYProperty().set(yStart);
        double posX = destWidth/2+(xStart-remainWidth-10);
        label.translateXProperty().set(posX);
    }

}
