package ch.sahits.game.openpatrician.javafx.control.skin;

import ch.sahits.game.graphic.image.IDataImageLoader;
import ch.sahits.game.openpatrician.javafx.control.ShipIcon;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.IGroupableVessel;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import javafx.beans.binding.DoubleBinding;
import javafx.geometry.Point2D;
import javafx.scene.Group;
import javafx.scene.control.Label;
import javafx.scene.control.SkinBase;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.paint.Color;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Nov 18, 2017
 */
@ClassCategory(EClassCategory.JAVAFX)
public class ShipIconSkin extends SkinBase<ShipIcon> {
    private static final double ICON_SCALE = 0.5;
    private static final double SHIELD_Y_OFFSET = 5;
    private final Logger logger = LogManager.getLogger(getClass());

    private final IDataImageLoader imageLoader;
    private final ShipIcon control;

    private Image shipIcon;
    private ImageView shipIconView;
    private ImageView shield;
    private ImageView piratingIndicator;
    private Label shipGroupSize;


    public ShipIconSkin(ShipIcon control, IDataImageLoader imageLoader) {
        super(control);
        this.imageLoader = imageLoader;
        this.control = control;
        initializeControl();
    }

    private void initializeControl() {
        Group g = new Group();
        INavigableVessel vessel = control.getVessel();
        shipIcon = getShipIcon(vessel);
        shipIconView = new ImageView(shipIcon);
        int directionFactor = control.isTravelingEast() ? -1 : 1;
        control.travelingEastProperty().addListener((observable, oldValue, newValue) -> {
            int dir = control.isTravelingEast() ? -1 : 1;
            shipIconView.setScaleX(ICON_SCALE * dir);
        });
        shipIconView.setScaleX(ICON_SCALE * directionFactor);
        shipIconView.setScaleY(ICON_SCALE);
        shield = createShield(vessel);
        shipGroupSize = new Label();
        if (vessel instanceof IGroupableVessel) {
            IGroupableVessel group = (IGroupableVessel) vessel;
            shipGroupSize.setText(String.valueOf(group.getShips().size()));
            DoubleBinding relYOrigin = shieldYOffsetBinding().add(2);
            shipGroupSize.layoutYProperty().bind(relYOrigin);
            shipGroupSize.layoutXProperty().bind(shipIconView.layoutXProperty().add(4));
            Color shieldColor = getShieldBaseColor(vessel);
            Color textColor = calculateContrastColor(shieldColor);
            shipGroupSize.setTextFill(textColor);
        }
        piratingIndicator = createPirateIndicator();
        g.getChildren().addAll(shipIconView, shield, shipGroupSize);
        if (vessel.getOwner().equals(control.getPlayer()) && vessel.getPirateFlag()) {
            g.getChildren().add(piratingIndicator);
        }
        vessel.pirateFlagProperty().addListener((observable, oldValue, newValue) -> {
            if (newValue) {
                // pirate flag was set:
                if (vessel.getOwner().equals(control.getPlayer())) {
                    g.getChildren().add(piratingIndicator);
                } else {
                    updateOtherPlayerShield(g, vessel);
                }
            } else { // priate flag was removed
                if (vessel.getOwner().equals(control.getPlayer())) {
                    g.getChildren().remove(piratingIndicator);
                } else {
                    updateOtherPlayerShield(g, vessel);
                }
            }
        });
        updateShipPosition();
        control.travelingEastProperty().addListener((observable, oldValue, newValue) -> updateShipPosition());
        control.scaleProperty().addListener((observable, oldValue, newValue) -> updateShipPosition());
        getChildren().add(g);
    }

    private void updateOtherPlayerShield(Group g, INavigableVessel vessel) {
        g.getChildren().remove(shield);
        shield = createShield(vessel);
        g.getChildren().add(shield);
        Color shieldColor = getShieldBaseColor(vessel);
        Color textColor = calculateContrastColor(shieldColor);
        shipGroupSize.setTextFill(textColor);
    }

    private Color calculateContrastColor(Color color) {
        double r = color.getRed();
        double g = color.getGreen();
        double b = color.getBlue();
        double gray = (r + g + b);
        if (gray > 0.7) {
            return Color.BLACK;
        } else {
            return Color.WHITE;
        }
    }

    public void updateShipPosition() {
        INavigableVessel vessel = control.getVessel();
        final Point2D location = vessel.getLocation();
        double x = location.getX() * control.scaleProperty().get();
        double y = location.getY() * control.scaleProperty().get();
        control.setLayoutX(x - shipIcon.getWidth() / 2);
        control.setLayoutY(y - shipIcon.getHeight() / 2);
        logger.trace("Update ship {} position to {},{} in view {},{}", vessel.getName(), vessel.getLocation().getX(), vessel.getLocation().getY(), control.getLayoutX(), control.getLayoutY());
    }

    private ImageView createShield(INavigableVessel vessel) {
        Image shieldImg = imageLoader.getImage("icons/Shield_" + getShieldColorName(vessel));
        ImageView imgView = new ImageView(shieldImg);
        imgView.setLayoutX(shipIconView.getLayoutX());
        DoubleBinding relYOrigin = shieldYOffsetBinding();
        imgView.layoutYProperty().bind(relYOrigin);
        imgView.layoutXProperty().bind(shipIconView.layoutXProperty());
        return imgView;
    }

    private String getShieldColorName(INavigableVessel vessel) {
        if (vessel.getOwner() instanceof IHumanPlayer) {
            return ((IHumanPlayer) vessel.getOwner()).getColor().name();
        } else if (vessel.getOwner() instanceof ISeaPirate
                || (!vessel.getOwner().equals(control.getPlayer())) && vessel.getPirateFlag()) {
            return "BLACK";
        } else {
            return "LIGHTGRAY";
        }
    }

    private Color getShieldBaseColor(INavigableVessel vessel) {
        Color color;
        if (vessel.getOwner() instanceof IHumanPlayer) {
            color = ((IHumanPlayer) vessel.getOwner()).getColor().getColor();
        } else if (vessel.getOwner() instanceof ISeaPirate
                || (!vessel.getOwner().equals(control.getPlayer())) && vessel.getPirateFlag()) {
            color = Color.BLACK;
        } else {
            color = Color.LIGHTGRAY;
        }
        return color;
    }

    private ImageView createPirateIndicator() {
        Image shieldImg = imageLoader.getImage("icons/pirateIndicator" );
        ImageView imgView = new ImageView(shieldImg);
        DoubleBinding relYOrigin = shieldYOffsetBinding();
        imgView.layoutYProperty().bind(relYOrigin);
        imgView.layoutXProperty().bind(shipIconView.layoutXProperty());
        return imgView;
    }

    private DoubleBinding shieldYOffsetBinding() {
        return shipIconView.layoutYProperty().add(shipIcon.getHeight() * ICON_SCALE - 0.5 * shipIcon.getHeight() * ICON_SCALE + SHIELD_Y_OFFSET);
    }

    /**
     * Load the ship icon for the appropriate type with size 64.
     * @param vessel
     * @return
     */
    private Image getShipIcon(INavigableVessel vessel) {
        if (vessel instanceof IShip) {
            switch (((IShip) vessel).getShipType()) {
                case COG:
                    return imageLoader.getImage("icons/64/cog_icon");
                case CRAYER:
                    return imageLoader.getImage("icons/64/crayer_icon");
                case HOLK:
                    return imageLoader.getImage("icons/64/holk_icon");
                case SNAIKKA:
                    return imageLoader.getImage("icons/64/schnikka_icon");
            }
        } else {
            IConvoy convoy = (IConvoy) vessel;
            return getShipIcon(convoy.getOrlegShip());
        }
        return null;
    }
}
