package ch.sahits.game.openpatrician.javafx.service;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.javafx.control.DecoratedText;
import ch.sahits.game.openpatrician.javafx.model.ENoticeBoardType;
import ch.sahits.game.openpatrician.javafx.model.NoticeBoardMenuEntry;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.l10n.Locale;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;

/**
 * Provide the menu items for the ship selection notice board menu.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 19, 2015
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class NoticeBoardShipSelectionMenuProvider implements INoticeBoardMenuProvider {
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private ShipService shipService;
    @Autowired
    protected DecoratedTextFactory textFactory;
    @Override
    public LinkedHashSet<NoticeBoardMenuEntry> createMenuEntries(ICityPlayerProxyJFX proxy, ENoticeBoardType type) {
        LinkedHashSet<NoticeBoardMenuEntry> entries = new LinkedHashSet<>();

        List<INavigableVessel> vessels = proxy.getPlayersNavalVessels();
        for (INavigableVessel vessel : vessels) {
            int weaponsStrength = shipService.calculateShipsWeaponsStrength(vessel);
            if (vessel instanceof IShip) {
                boolean disabled = false;
                final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.ship";
                NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, vessel, disabled, key,false, 200);
                entries.add(entry);
            } else {
                // it is a convoy
                IConvoy convoy = (IConvoy) vessel;
                boolean disabled = !convoy.getOwner().equals(proxy.getPlayer());
                if (weaponsStrength < 5) {
                    final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.convoy.captain.smallstrangth.capacity";
                    NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, convoy, disabled, key, false, 160);

                    entries.add(entry);
                } else {
                    final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.convoy.captain.bigstrangth.capacity";
                    NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, convoy, disabled, key, false, 160);
                    entries.add(entry);
                    }
                List<IShip> convoyMembers = convoy.getShips();
                for (IShip convoyMember : convoyMembers) {
                    disabled = !convoyMember.getOwner().equals(proxy.getPlayer());
                    final String key = "ch.sahits.game.javafx.service.NoticeBoardShipSelectionMenuProvider.shipinconvoy";
                    NoticeBoardMenuEntry entry = createNoticeBoardEntry(proxy, convoyMember, disabled, key, true, 180);
                    entries.add(entry);
                }
            } // end else convoy
        }
        return entries;
    }

    private NoticeBoardMenuEntry createNoticeBoardEntry(ICityPlayerProxyJFX proxy, INavigableVessel vessel, boolean disabled, String key, boolean inset, int wrappingWidth) {
        String loadedText = messageSource.getMessage(key,
                new Object[]{vessel.getName(), vessel.getCapacity()}, locale.getCurrentLocal());
        return createMenuEntry(proxy, vessel, loadedText, disabled, wrappingWidth);
    }

    private NoticeBoardMenuEntry createMenuEntry(ICityPlayerProxyJFX proxy, INavigableVessel vessel, String loadedText, boolean disabled, int wrappingWidth) {
        DecoratedText dt = textFactory.createDecoratedText(loadedText, new HashMap<>());
        dt.setWrappingWidth(210);
        NoticeBoardMenuEntry entry = new NoticeBoardMenuEntry(null);
        entry.setLabel(dt);
        entry.setDisabled(disabled);
        entry.setSelected(vessel.equals(proxy.getActiveShip()));
        return entry;
    }


}
