package ch.sahits.game.openpatrician.model;

import java.io.IOException;
import java.util.Calendar;
import java.util.Random;

import ch.sahits.game.openpatrician.model.building.BuildingFactory;
import ch.sahits.game.openpatrician.model.city.CityFactory;
import ch.sahits.game.openpatrician.model.city.ECityName;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.impl.PersonalData;
import ch.sahits.game.openpatrician.model.time.DateObject;

public class GameFactory {
	private static final Random rnd = new Random(System.nanoTime());

	/**
	 * Create a human player instance. Build their trading office in the home town as well.
	 * @param name
	 * @param lastName
	 * @param homeTown
	 * @param male
	 * @return
	 */
	public static IPlayer createPlayer(String name, String lastName, ICity homeTown, boolean male, int cash) {
		int age = rnd.nextInt(10)+17;
		PersonalData pd = new PersonalData(name, lastName, male, homeTown, computeRandomBirthDate(age));
		IPlayer player = new Player( homeTown, pd,cash,ESocialRank.CHANDLER);
		BuildingFactory.createTridingOffice(player, homeTown,0); // initial trading office has no value
		initPlayerInCities(player);
		return player;
	}
	/**
	 * Create an artifiacial player instance. Build their trading office in the home town as well.
	 * @param homeTown
	 * @return
	 */
	public static IPlayer createAIPlayer(ICity homeTown,long cash) {
		int age = rnd.nextInt(15)+17;
		IPlayer player = new AIPlayer(homeTown,cash,computeRandomBirthDate(age));
		BuildingFactory.createTridingOffice(player, homeTown,0); // initial trading office has no value
		initPlayerInCities(player);
		return player;
	}
	/**
	 * Init the player data in all cities
	 * @param player
	 */
	private static void initPlayerInCities(IPlayer player) {
		ECityName[] cityNames = ECityName.values();
		for (int i = 0; i < cityNames.length; i++) {
			try {
				ICity city =CityFactory.createCityByName(cityNames[i]);
				city.moveIn(player);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
	/**
	 * Create the standard map with all cities defined in {@link ECityName}
	 * @return standard map of the game
	 * @throws IOException Cities properties could not be read
	 */
	public static IMap createMap() throws IOException{
		ECityName[] cityNames = ECityName.values(); // TODO ther might be customizable city names
		ICity[] cities = new ICity[cityNames.length];
		for (int i = 0; i < cityNames.length; i++) {
			cities[i]=CityFactory.createCityByName(cityNames[i]);
		}
		return new GameMap(cities);
	}
	public static IGame createGame(IMap map, EObjective objective, Difficulty difficulty, EGameSpeed speed, int startYear){
		return new Game(map, objective, difficulty, speed, Date.getInstance(startYear));
	}
	
	private static DateObject computeRandomBirthDate(int age){
		Date date = Date.getInstance(0); // ugly
		int year = date.getStartYear()-age;
		int month = rnd.nextInt(12)+1;
		int day;
		if (month==2){
			day = rnd.nextInt(28)+1;
		} else if (month==4 || month==6 || month==9 || month==11){
			day = rnd.nextInt(30)+1;
		} else {
			day = rnd.nextInt(31)+1;
		}
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, year);
		cal.set(Calendar.MONTH, month);
		cal.set(Calendar.DAY_OF_MONTH, day);
		return new DateObject(cal);
	}
}
