package ch.sahits.game.openpatrician.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.impl.Company;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.util.ClassChecker;

class Player implements IPlayer{

	private final LinkedList<IShip> fleet;
	private ESocialRank rank; // TODO this has to be updated monthly
	private ICareer careerLevel; // TODO this has to be updated
	private IPersonalData personalData;
	private ISpouseData spouse;
	private final ICompany company;
	
	
	public Player(ICity homeTown, IPersonalData personalData, long cash,ESocialRank rank) {
		super();
		fleet = new LinkedList<IShip>();
		this.rank=rank;
		careerLevel=EEconomicCareer.INEXPERIANCED;
		this.personalData=personalData;
		company = new Company(this, homeTown, cash);
	}

	@Override
	public String getName() {
		return personalData.getName();
	}

	@Override
	public String getLastName() {
		return personalData.getLastName();
	}

	@Override
	public ICity getHometown() {
		return company.getHomeTown();
	}

	@Override
	public List<IShip> getFleet() {
		return Collections.unmodifiableList(fleet);
	}

	@Override
	public boolean isMale() {
		return personalData.isMale();
	}

	@Override
	public void addShip(IShip ship) {
		fleet.add(ship);
	}

	@Override
	public void removeShip(IShip ship) {
		fleet.remove(ship);
	}

	@Override
	public long getCash() {
		return company.getCash();
	}
	/**
	 * Add or subtract some cash
	 * @param diff amount of money that is transferred
	 */
	@Override
	public void updateCash(int diff){
		company.updateCash(diff);
	}

	@Override
	public List<IBuilding> findBuildings(ICity city) {
		ArrayList<IBuilding> result = new ArrayList<IBuilding>();
		for (IBuilding building : city.getBuildings()) {
			if (building.getOwner()==this){
				result.add(building);
			}
		}
		return result;
	}

	@Override
	public ITradingOffice findTradingOffice(ICity city) {
		for (IBuilding building : city.getBuildings()) {
			if (building.getOwner()==this && building instanceof ITradingOffice){
				return (ITradingOffice) building;
			}
		}
		return null;
	}

	@Override
	public List<IBuilding> findBuildings(ICity city,Class<? extends IBuilding> buildingClass) {
		List<IBuilding> result = new ArrayList<IBuilding>();
		for (IBuilding buidlding : city.getBuildings()) {
			ClassChecker checker = new ClassChecker(buidlding.getClass());
			if (checker.extendsClass(buildingClass) || checker.implementsInterface(buildingClass)){
				result.add(buidlding);
			}
		}
		return result;
	}

	@Override
	public ESocialRank getRank() {
		return rank;
	}

	@Override
	public ICareer getCareerLevel() {
		return careerLevel;
	}
	/**
	 * Marry a spouse
	 * @param spouse
	 */
	public void marry(ISpouseData spouse){
		this.spouse=spouse;
	}
	/**
	 * Widow and become unmarried again
	 */
	public void widow(){
		spouse=null;
	}

	@Override
	public IPersonalData getPersonalData() {
		return personalData;
	}

	@Override
	public ISpouseData getSpouseData() {
		return spouse;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((personalData == null) ? 0 : personalData.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Player other = (Player) obj;
		if (personalData == null) {
			if (other.personalData != null)
				return false;
		} else if (!personalData.equals(other.personalData))
			return false;
		return true;
	}

	@Override
	public ICompany getCompany() {
		return company;
	}

	@Override
	public void updateRank(ESocialRank rank) {
		this.rank=rank;
	}

}
