package ch.sahits.game.openpatrician.model.building.impl;

import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ELevel;
import ch.sahits.game.openpatrician.model.building.IStorage;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWarehouse;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.time.DateObject;
import ch.sahits.game.openpatrician.model.time.EUpdateIntervalRegistration;
import ch.sahits.game.openpatrician.model.time.IPeriodicalUpdate;
import ch.sahits.game.openpatrician.model.time.PeriodicalTimeUpdater;

public class Storage implements IStorage, IPeriodicalUpdate {
	
	public static final int GUARD_COST_PER_DAY = 10;
	public static final int RENT_COST_PER_DAY_AND_BALE = 20;
	
	private final IPlayer owner;
	private final ICity city;
	
	private int rentOutStorage;
	private int nbGuards;

	public Storage(IPlayer owner, ICity city) {
		super();
		this.owner = owner;
		this.city = city;
		new PeriodicalTimeUpdater(EUpdateIntervalRegistration.DAY, this);
	}
	/**
	 * Retrieve the number of warehouses
	 * @return
	 */
	private int getNbWareHouses(){
		return owner.findBuildings(city, IWarehouse.class).size();
	}

	@Override
	public int getRentOutStorage() {
		return rentOutStorage;
	}

	@Override
	public void updateRendedSpace(int nbBarrels) {
		if (-nbBarrels>rentOutStorage){
			rentOutStorage=0;
		} else {
			rentOutStorage += nbBarrels;
		}

	}

	@Override
	public int getCostsPerDay() {
		ITradingOffice office = owner.findTradingOffice(city);
		int additional = office.getCapacity()-office.getStored();
		if (additional>=0) return 0; // empty space
		return (-additional/10+1)*RENT_COST_PER_DAY_AND_BALE;
	}

	@Override
	public int getNumberGuards() {
		return nbGuards;
	}

	@Override
	public int getGuardCostsPerDay() {
		return nbGuards*GUARD_COST_PER_DAY;
	}

	@Override
	public ELevel getSecurityLevel() {
		double nb = getNbWareHouses();
		double factor =nbGuards/(nb+1);
		if (factor>=1) return ELevel.MAXIMUM;
		if (factor>=0.66) return ELevel.HIGH;
		if (factor<=0.33) return ELevel.LOW;
		return ELevel.MEDIUM;
	}
	@Override
	public void updateGuardsNumber(int update) {
		if (-update>nbGuards){ // reduce by more than are available
			nbGuards=0;
		} else {
			nbGuards += update;
		}
		
	}
	@Override
	public void notify(DateObject date) {
		int costs = getCostsPerDay()+getGuardCostsPerDay();
		owner.updateCash(-costs);
	}

}
