package ch.sahits.game.openpatrician.model.city;

import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;

import ch.sahits.game.openpatrician.model.city.impl.Bremen;
import ch.sahits.game.openpatrician.model.city.impl.Cologne;
import ch.sahits.game.openpatrician.model.city.impl.Danzig;
import ch.sahits.game.openpatrician.model.city.impl.Hamburg;
import ch.sahits.game.openpatrician.model.city.impl.Luebeck;
import ch.sahits.game.openpatrician.model.city.impl.Rostock;
import ch.sahits.game.openpatrician.model.city.impl.Stettin;
import ch.sahits.game.openpatrician.model.city.impl.Visby;
/**
 * Factory class for the cities of the game.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
public class CityFactory {
	/**
	 * Cache storing the already created cities to  avoid duplicate creation
	 */
	private static ConcurrentHashMap<String,ICity> createdCityCache = new ConcurrentHashMap<String,ICity>();
	/**
	 * Create a standard city based on its name
	 * @param cityName
	 * @return Instance of the city
	 * @throws IOException Error reading the city properties
	 */
	public static ICity createCityByName(String cityName) throws IOException{
		ECityName name = ECityName.valueOf(cityName.toUpperCase());
		return createCityByName(name);
	}
	/**
	 * Create a standard city based on its unique identification
	 * @param cityName
	 * @return Instance of the city
	 * @throws IOException Error reading the city properties
	 */
	public static ICity createCityByName(ECityName cityName) throws IOException {
		return createIfNotStroed(cityName);
	}
	/**
	 * Retrieve the city from the cache or create the city and put it there.
	 * @param name city name should not be null.
	 * @return city instance of null if the <code>name</code> is null
	 * @throws IOException Error reading the city properties
	 */
	private static ICity createIfNotStroed(ECityName name) throws IOException{
		if (!createdCityCache.containsKey(name.name())){
			ICity city;
			switch (name){
			case BREMEN:
				city = new Bremen();
				break;
			case DANZIG:
				city = new Danzig();
				break;
			case HAMBURG:
				city = new Hamburg();
				break;
			case KOELN:
				city = new Cologne();
				break;
			case LUEBECK:
				city = new Luebeck();
				break;
			case ROSTOCK:
				city = new Rostock();
				break;
			case STETTIN:
				city = new Stettin();
				break;
			case VISBY:
				city = new Visby();
				break;
			default:
				throw new IllegalArgumentException(name+" is not an implemented city");
			}
			createdCityCache.put(name.name(), city);
			return city;
		} else {
			return createdCityCache.get(name.name());
		}
	}
}
