package ch.sahits.game.openpatrician.model.product;

import java.io.IOException;
import java.util.Properties;
import java.util.ResourceBundle;

import ch.sahits.game.openpatrician.model.city.ECityState;
import ch.sahits.game.openpatrician.model.city.IPopulationStructure;
import ch.sahits.game.openpatrician.util.PropertyLoader;
import ch.sahits.game.openpatrician.util.l10n.Locale;
/**
 * Define the different type of wares that are produced and can be traded.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 20, 2011
 *
 */
public enum EWare implements IWare, IPriceCalculation {
	/*
	 * The order of the enumation is important
	 * as it is the order as it is in the dialogs.
	 */
	/** Beer in barrels */
	BEER("beer.properties"),
	/** Iron ore in burden */
	IRONORE("ironore.properties"),
	/** Iron in barrels */
	IRON("iron.properties"),
	/** Fur in barrels */
	FUR("fur.properties"),
	/** Fish in burden */
	FISH("fish.properties"),
	/** Meat in burden */
	MEAT("meat.properties"),
	/** Grain in burden */
	GRAIN("grain.properties"),
	/** Spices in barrels */
	SPICE("spice.properties"),
	/** Hemp in burden */
	HEMP("hemp.properties"),
	/** Wood in burden */
	WOOD("wood.properties"),
	/** Honey in barrels */
	HONEY("honey.properties"),
	/** Pottery in barrels */
	POTTERY("pottery.properties"),
	/** Leather in barrels */
	LEATHER("leather.properties"),
	/** Pitch in barrels */
	PITCH("pitch.properties"),
	/** Salt in barrels */
	SALT("salt.properties"),
	/** Fish oil in barrels */
	FISH_OIL("fishoil.properties"),
	/** Cloth in barrels */
	CLOTH("cloth.properties"),
	/** Wine in barrels */
	WINE("wine.properties"),
	/** Wool in burden */
	WOOL("wool.properties"),
	/** Brick in burden */
	BRICK("brick.properties");
	/*
	 * For now the algorithm for the price is implemented here
	 * this means the price development is always the same.
	 */

	private final Locale locale = Locale.getInstance();
	private final EWareSize size;
	private final int maxValueBuy;
	private final int minValueBuy;
	private final int maxValueSell;
	private final int minValueSell;
	private final String wareID;
	private final int saturation;
	private final IPriceCalculation calculation = new SimplePriceCalculation();
	/**
	 * Initialize the ware based on a configuration file
	 * @param configFileName file name of the configuration property file
	 * @throws IOException if the configuration file could not be read or properties are missing.
	 */
	private EWare(String configFileName){
		Properties props = PropertyLoader.loadProperties(configFileName);
		if (props.getProperty("type")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the type property");
		}
		if (props.getProperty("sizeUnit")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the sizeUnit property");
		}
		if (props.getProperty("maxValueBuy")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the maxValueBuy property");
		}
		if (props.getProperty("minValueBuy")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the minValueBuy property");
		}
		if (props.getProperty("maxValueSell")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the maxValueSell property");
		}
		if (props.getProperty("minValueSell")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the minValueSell property");
		}
		if (props.getProperty("localisationID")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the localisationID property");
		}
		if (props.getProperty("saturation")==null){
			throw new IllegalArgumentException("The property file "+configFileName+" does not contain the saturation property");
		}
		size=EWareSize.valueOf(props.getProperty("sizeUnit"));
		maxValueBuy=Integer.parseInt(props.getProperty("maxValueBuy"));
		minValueBuy=Integer.parseInt(props.getProperty("minValueBuy"));
		wareID=props.getProperty("localisationID");
		saturation=Integer.parseInt(props.getProperty("saturation"));
		minValueSell=Integer.parseInt(props.getProperty("minValueSell"));
		maxValueSell=Integer.parseInt(props.getProperty("maxValueSell"));
		
	}
	
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getLocalDisplayName()
	 */
	@Override
	public final String getLocalDisplayName(){
		return getLocalDisplayName(wareID);
	}
	/**
	 * Retrieve the localized display name for <code>id</code>.
	 * @param id
	 * @return
	 */
	private String getLocalDisplayName(String id){
		ResourceBundle messages = ResourceBundle.getBundle("ModelMessages", locale.getCurrentLocal());
		return messages.getString(id);
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getMaxValueBuy()
	 */
	@Override
	public int getMaxValueBuy() {
		return maxValueBuy;
	}
	/**
	 * Retrieve the highest price of this ware when only on item is available
	 * @return
	 */
	@Override
	public int getMaxValueSell() {
		return maxValueSell;
	}
	/**
	 * Retrieve the lowest price this ware is soled (at market saturation)
	 * @return
	 */
	@Override
	public int getMinValueSell() {
		return minValueSell;
	}

	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getMinValueBuy()
	 */
	@Override
	public int getMinValueBuy() {
		return minValueBuy;
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#computeSellPrice(int, int)
	 */
	public int computeSellPrice(int availableAmount, int amountToSell) {
		if (amountToSell<=0){
			throw new IllegalArgumentException("The amount to sell must be larger than 0 (was "+availableAmount+") for "+name());
		}
		if (availableAmount<0){
			throw new IllegalArgumentException("The available amount must be positive for "+name());
		}
		if (availableAmount>=getMarketSaturation()){
			return getMinValueSell();
		}
		int min = getMinValueSell();
		int max = getMaxValueSell();
		// Take amounts
		int firstItemPrice = (int)Math.rint(computePrice(min, max, availableAmount,getMarketSaturation(),0,null,null));
		if (amountToSell>1){
			int lastItemPrice =  (int)Math.rint(computePrice(min, max, availableAmount+amountToSell,getMarketSaturation(),0,null,null));
			return (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		} else {
			return firstItemPrice;
		}
	}
//	/**
//	 * Compute the price for one item of the ware.
//	 * @param min the lowest possible price (0 exclusive)
//	 * @param max the highest possible price
//	 * @param available amount of wares that are available in the market.
//	 * @return price for one item sold
//	 */
//	protected int computePrice(int min, int max, int available) {
//		//FIXME: this does not produce correct values in between:
//		// (MAX-MIN)*(MIN+1)/(exp(0.5*x*10/SATURATION*1.5)+MIN)+ABS_MIN
//		// TODO: improve the argument of exp so the MIN value is achieved at SATURATION
//		// TODO use an implementation of IPriceCalculation
//		int marketSaturation = getMarketSaturation();
//		double scale = marketSaturation*1.5/10;
//		return (int) Math.rint((max-min)*(min+1)/(Math.exp(0.5/scale)+min))+getMinValueSell();
//	}
	/**
	 * {@inheritDoc}
	 * @see SimplePriceCalculation#computePrice(int, int, int, int, int, IPopulationStructure, ECityState)
	 */
	@Override
	public double computePrice(int min, int max, int available, int saturation,
			int productionRate, IPopulationStructure pop, ECityState state) {
		return calculation.computePrice(min, max, available, saturation, productionRate, pop, state);
	}


	@Override
	public int computeBuyPrice(int availableAmount, int amountToBuy) {
		if (amountToBuy<=0){
			throw new IllegalArgumentException("The amount to buy must be larger than 0 for "+name());
		}
		if (availableAmount<0){
			throw new IllegalArgumentException("The available amount must be positive for "+name());
		}
		if (availableAmount==0){
			return 0;
		}
		if (availableAmount>=getMarketSaturation()){
			return getMinValueBuy();
		}
		int min = getMinValueBuy();
		int max = getMaxValueBuy();
		// Take amounts
		int firstItemPrice =  (int)Math.rint(computePrice(min, max, availableAmount,getMarketSaturation(),0,null,null));
		if (amountToBuy>1){
			// TODO make sure the value is not above the limit
			int lastItemPrice =  (int)Math.rint(computePrice(min, max, availableAmount-amountToBuy,getMarketSaturation(),0,null,null));
			return (int) Math.rint((firstItemPrice+lastItemPrice)/2.0);
		} else {
			return firstItemPrice;
		}
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getSizeAsBarrels()
	 */
	@Override
	public final short getSizeAsBarrels(){
		return size.getRealMasurement();
	}
	/* (non-Javadoc)
	 * @see ch.sahits.game.openpatrician.model.product.IWare#getMarketSaturation()
	 */
	@Override
	public int getMarketSaturation() {
		return saturation;
	}

}
