package ch.sahits.game.openpatrician.model.product;

import ch.sahits.game.openpatrician.model.IAmountable;

/**
 * Interface defining a ware. The ware provides general information on
 * the item it represents. It also provides methods to compute
 * the sell and buy price of the ware based on the available amount.
 * The price computation uses the following constants:<br/>
 * <code>MAX</code>: the maximum price for the item if it is not available in the market.
 * Here the function has a glitch for the buy price: If no item is available
 * it cannot be sold and the price is therefore 0 (infinity would be logical correct
 * but is not very understandable in the game context).<br>
 * <code>MIN</code>: the minimal price of the item if more than <code>SATURATION</code> items
 * are available in the market. As more items are available in the market the price approaches
 * this value asymptoticly.<br/>
 * <code>SATURATION</code>: amount of items when the the need is fullfilled. If there are more items
 * available in the market than needed all have the <code>MIN</code> price. This is a further
 * simplification because the price normally drops further if there are much more items available.
 * The curve would bee leveling out at <code>SATURATION</code> and then dropping further.<br/>
 * <code>ABS_MIN</code>: the absolut minimum is the minimal sell price at <code>SATURATION</code>.<br/>
 * <pre>
 *   (MAX-MIN)*(MIN+1)
 * --------------------- + ABS_MIN
 * e<sup>x*15/(SATURATION*2)</sup>+MIN
 * </pre>
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jan 15, 2011
 *
 */
public interface IWare extends IAmountable{

	public abstract String getLocalDisplayName();

	/**
	 * Retrieve the value that is maximal payed for one unit of this ware
	 * if no ware is available.
	 * @return
	 */
	public abstract int getMaxValueBuy();

	/**
	 * Retrieve the value of the minimal payed sum for on unit if an infinitly amount
	 * of the ware exists.
	 * @return
	 */
	public abstract int getMinValueBuy();

	/**
	 * Compute the price for the wares to be soled. The price is computed based on the amount
	 * that is available and the amount that is sold. The price is never higher than {@link #getMaxValueBuy()}
	 * and approaches {@link #getMinValueBuy()} assymtotically.
	 * @param availableAmount Amount of the ware that is available
	 * @param amountToSell Amount of ware that is to be sold
	 * @return price of the ware.
	 */
	public abstract int computeSellPrice(int availableAmount, int amountToSell);

	/**
	 * Compute the price for the wares to be bought. The price is computed on the amount
	 * that is available and the amount to be bought. The price is never highter than {@link #getMaxValueBuy()}
	 * plus a ware specific constant.
	 * @param availableAmount Amount of the ware that is available
	 * @param amountToBuy Amount of ware that is to be bought
	 * @return price of the ware
	 */
	public abstract int computeBuyPrice(int availableAmount, int amountToBuy);

	/**
	 * Retrieve the size of the ware converted into the smallest size unit (barrels)
	 * @return
	 */
	public abstract short getSizeAsBarrels();

	/**
	 * Retrieve the amount of units which is considered infinite. This influences the price because
	 * if there are so much units of the ware available the selling price will be the minimum.
	 * @return
	 */
	public abstract int getMarketSaturation();

	public abstract int getMinValueSell();

	public abstract int getMaxValueSell();
	/**
	 * As this interface is implemented by an enumeration add this method
	 * @return
	 */
	public String name();

}