package ch.sahits.game.openpatrician.model.time;

import java.util.Calendar;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IClockUpdateListener;
/**
 * The periodical time updater encapsules clock ticks of instances that are interested in
 * notification but only on a periodical level. The period may vary but the implementation is
 * always the same. The periods as defined in {@link EUpdateIntervalRegistration} are possible,
 * where as a year is composed by 365 days and a month has always 30 days. So these periods are
 * not really exact.
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 22, 2012
 *
 */
public class PeriodicalTimeUpdater implements IClockUpdateListener{
	/** Period of the clock ticking */
	private int tickInterval;
	private final Date date;
	/** Period that the listener should be notified */
	private final int updateInterval;
	/** Time passed since the last notification */
	private int timePassed;
	/** Instance that is to be notified */
	private IPeriodicalUpdate listener;
	public PeriodicalTimeUpdater(EUpdateIntervalRegistration intervall, IPeriodicalUpdate listener){
		date = Date.getInstance(0); // TODO ugly
		tickInterval = date.getUpdateInterval();
		this.listener=listener;
		date.add(this);
		final Calendar cal = date.getCurrentDate();
		DateObject dateO = new DateObject(cal);
		switch (intervall) {
		case YEAR:
			timePassed = (cal.get(Calendar.DAY_OF_YEAR)-1)*24*60;
			timePassed += dateO.getHour()*60;
			timePassed += dateO.getMinute();
			updateInterval = 24*60*365;
			break;
		case MONTH:
			timePassed = (dateO.getDayInMonth()-1)*24*60;
			timePassed += dateO.getHour()*60;
			timePassed += dateO.getMinute();
			updateInterval = 24*60*30;
			break;
		case WEEK:
			timePassed = dateO.getDayInWeek()*24*60;
			timePassed += dateO.getHour()*60;
			timePassed += dateO.getMinute();
			updateInterval = 7*24*60;
			break;
		case DAY:
			timePassed = dateO.getHour()*60;
			timePassed += dateO.getMinute();
			updateInterval = 24*60;
			break;
			// TODO add last of month
		default:
			throw new IllegalArgumentException("Unknown update interval: "+intervall);
		}
	}
	/**
	 * Stop notifiyng the listener
	 */
	public void stop(){
		listener=null;
	}

	@Override
	public void updateTick(int minutes) {
		tickInterval = minutes;
	}

	@Override
	public void ticked() {
		timePassed += tickInterval;
		if (timePassed>updateInterval){
			if (listener!=null){
				listener.notify(new DateObject(date.getCurrentDate()));
			}
			timePassed=0;
		}
	}

}
