package ch.sahits.game.openpatrician.model;

import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import lombok.Getter;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import java.util.Calendar;

/**
 * Representation of the date within the game.
 * This class is thread safe.
 * The Date is implemented as a singleton. However there
 * is no getInstance method to retrieve the once created instance,
 * so it must be referenced elsewhere.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 16, 2011
 *
 */
@Component
public class Date {
	/** Tick update in minutes */
	@Getter
    private int tickUpdate;
	/** Current date */
	private ObjectProperty<DateTime> cal = new SimpleObjectProperty<>(this, "cal", null);;
	@Getter
    private DateTime startDate;
	/** Start year */
    @Getter
	private Integer startYear = null;
	/** Lock for guaranteeing thread safety */
	private static Object lock = new Object();
    private StringBinding dayDateBinding = null;
	@Autowired
	@Qualifier("serverClientEventBus")
	private AsyncEventBus clientServerEventBus;
	@Autowired
	private Locale locale;
	@Autowired
	private MessageSource messageSource;
	/**
	 * Initialize the date with start year.
	 * The initial date is 13th July of the start year.
	 * The default tick update is 5 minutes.
	 */
	public Date(int year){
		this();
		setStartYear(year);
	}


	public Date() {
		tickUpdate = 5;
	}
	public void setStartYear(int startYear) {
		Preconditions.checkState(this.startYear == null, "Date may only be initialized once");
		System.out.println("+++++++++++++++++++ set start year: "+startYear);
		this.startYear=startYear;
		updateTime(new DateTime(startYear, 7, 13, 0, 0));
		startDate = getCurrentDate();
	}
	/**
	 * Constructor for testing purposes only
	 * @param cal
	 */
	Date(DateTime cal){
		updateTime(cal);
		startYear = cal.getYear();
		tickUpdate = 5;
	}

    /**
     * Change the time.
     * @param newTime
     */
	public void updateTime(DateTime newTime) {
		Preconditions.checkNotNull(newTime);
		cal.set(newTime);
	}

	/**
	 * Set the tick update in number of minutes
	 * @param minutes tick update
	 */
	public void setTickUpdate(int minutes){
		tickUpdate =minutes;
		final ClockTickIntervalChange event = new ClockTickIntervalChange();
		event.setInterval(tickUpdate);
		clientServerEventBus.post(event);
	}

	/**
	 * Retrieve the date of the start of the week in the form {Day of month}. {Month}.
	 * The week starts with Monday
	 * @return
	 */
	public String getStartOfWeek(){
		DateTime lastMon = getStartOfWeekInternal();
		return toShortDate(lastMon);
	}
	/**
	 * Retrieve the date of the start of the last week
	 * @return
	 */
	DateTime getStartOfWeekInternal() {
		// last week
		DateTime lastMonday = cal.get().minusWeeks(1);
		// first day
		lastMonday = lastMonday.minusDays(lastMonday.getDayOfWeek()-1);
		return lastMonday;
	}
	/**
	 * Convert the date into the form {Day in month}. {Month}
	 * @param c
	 * @return
	 */
	private String toShortDate(DateTime c) {
		final int day;
		final int month;
		synchronized (lock) {
			day = c.getDayOfMonth();
			month = c.getMonthOfYear();
		}
		StringBuilder sb = new StringBuilder();
		sb.append(day).append(". ");
		sb.append(messageSource.getMessage("month_short_"+month, new Object[]{}, locale.getCurrentLocal()));

		return sb.toString();
	}
	/**
	 * Retrieve the date of the end of the week in the form {Day of month}. {Month}.
	 * The week ends with Sunday
	 * @return
	 */
	public String getEndOfWeek(){
		DateTime lastSun = getEndOfWeekInternal();
		return toShortDate(lastSun);
	}
	/**
	 * Retrieve the date of the of the Sunday of last week
	 * @return
	 */
	DateTime getEndOfWeekInternal() {
		int cdow = cal.get().getDayOfWeek();
		DateTime lastDayOfWeek = cal.get().plusDays(7 - cdow);
		return lastDayOfWeek;
	}

	/**
	 * Create a string representation for the user interface of the
	 * form {Day of month}. {Month} {Year}.
	 * @return
	 */
	public String toDisplayString(){
		final int day;
		final int month;
		final int year;
		synchronized (lock) {
			day = cal.get().getDayOfMonth();
			month = cal.get().getMonthOfYear();
			year = cal.get().getYear();
		}
		return todisplayString(day, month, year);
	}

    /**
     * Bind the representation of the day as a String.
     * @return
     */
    public StringBinding dayDateBinding() {
          if (dayDateBinding == null) {
              dayDateBinding = new StringBinding() {
                  @Override
                  protected String computeValue() {
                      return toDisplayString();
                  }
              };
          }
        return dayDateBinding;
    }
	// TODO aho Jan 28, 2013: add utility with DateTime and formatter
	public String todisplayString(final int day, final int month, final int year) {
		StringBuilder sb = new StringBuilder();
		sb.append(day).append(". ");
		sb.append(messageSource.getMessage("month_short_" + month, new Object[]{}, locale.getCurrentLocal())).append(" ");
		sb.append(year);
		return sb.toString();
	}
	/**
	 * Retrieve the immutable DateTime
	 * @return
	 */
	public DateTime getCurrentDate(){
		return cal.get();
	}
	/**
	 * Retrieve the immutable DateTime
	 * @return
	 */
	public ObjectProperty<DateTime> getCurrentDateProperty(){
		return cal;
	}
	/**
	 * Retrieve an index for the current weekday. Mondy is 0.
	 * @return
	 */
	public final int getWeekdayIndex(){
		int dayOfWeek = cal.get().getDayOfWeek()+1;
		return (dayOfWeek-Calendar.MONDAY)%7;
	}
}
