package ch.sahits.game.openpatrician.model.city.impl;

import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.collections.NonReplacableMap;
import ch.sahits.game.openpatrician.data.map.Production;
import ch.sahits.game.openpatrician.javafx.bindings.LateIntegerBinding;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.city.EKontorType;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.factory.PlayerInteractionFactory;
import ch.sahits.game.openpatrician.model.impl.WareHolding;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.personal.impl.Reputation;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.PropertyLoader;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.geometry.Point2D;
import lombok.Getter;
import lombok.Setter;
import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.ResourceBundle;

/**
 * Implementation of the city model. The model of the city should only be instanciated once.
 * A city is unique. therefore equality can be tested by identity.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Jan 18, 2011
 *
 */
@Prototype
public class City extends WareHolding implements ICity {
    private  final Logger logger = Logger.getLogger(getClass());
	@Autowired
	private Random rnd;
	// TODO aho Jan 25, 2013: autowire rnd
	// TODO set the polulation structure based on the settings from the properties and the start year
	// TODO aho Jan 24, 2013: split up the class to separate model data from operations
    @Getter
	private final IWare[] effectiveProduction;
    @Getter
	private final IWare[] ineffectiveProduction;
    @Getter
	private String name;
	private final String nameTemplate;
    @Getter
	private final EKontorType kontorType;
	@Autowired
	private PlayerInteractionFactory interactionFactory;
	@Autowired
	private Locale locale;
	@Autowired
	private MessageSource messageSource;
	@Getter
	private final Point2D coordinates;
	private final Map<IWare,DateTime> missingWares = new HashMap<IWare, DateTime>();
	//	/** Store the amount of wares in the city in the ware specific sizes */
	//	final HashMap<IWare, AmountablePrice> wares = new HashMap<IWare, AmountablePrice>();
	/** Store the buildings in the city */
	private List<IBuilding> buildings = new ArrayList<IBuilding>();
	/** Map holding the reputation of the different players */
	private Map<IPlayer,IReputation> reputation = new HashMap<IPlayer, IReputation>();
	/** Store the contibutions of the players */
	private Map<IPlayer,Contributions> playersContributions = new HashMap<IPlayer, Contributions>();
	/** Holding the population split by population classes */
	private final Map<EPopulationClass, IntegerProperty> population = new NonReplacableMap<>();
	
	private LateIntegerBinding totalPopulation = new LateIntegerBinding() {
		
		@Override
		protected int computeValue() {
			return internalPopulationCount();
		}
	};
    @Getter
    @Setter
	private CityState cityState;
	@Autowired
	private Date date;

    public City(ch.sahits.game.openpatrician.data.map.City city) {
        Production production = city.getProduction();
		Preconditions.checkNotNull(city.getName(), "The city does not define a name");
		nameTemplate = city.getName();
		Preconditions.checkArgument(!production.getEffective().getWare().isEmpty(), "The city "+nameTemplate+" does not contain the effectiveProduction property");
		Preconditions.checkArgument(!production.getIneffective().getWare().isEmpty(), "The city "+nameTemplate+" does not contain the ineffectiveProduction property");
		Preconditions.checkNotNull(city.getKontorType(), "The city "+nameTemplate+" does not define the kontor type.");
		Preconditions.checkNotNull(city.getLocation().getX(), "The city "+nameTemplate+" does not contain the city location");
		Preconditions.checkNotNull(city.getLocation().getY(), "The city "+nameTemplate+" does not contain the city location");
        effectiveProduction = new EWare[production.getEffective().getWare().size()];
        for (int i = 0; i < production.getEffective().getWare().size(); i++) {
            effectiveProduction[i]=EWare.valueOf(production.getEffective().getWare().get(i));
        }
        ineffectiveProduction = new EWare[production.getIneffective().getWare().size()];
        for (int i = 0; i < production.getIneffective().getWare().size(); i++) {
            ineffectiveProduction[i]=EWare.valueOf(production.getIneffective().getWare().get(i));
        }
        kontorType = EKontorType.valueOf(city.getKontorType());
        int x = city.getLocation().getX();
        int y = city.getLocation().getY();
        coordinates = new Point2D(x, y);
	}
	@PostConstruct
	private void init() {
		this.name=messageSource.getMessage(nameTemplate, new Object[0], locale.getCurrentLocal());
		initWares();
		initPopulation();
	}

	/**
	 * This Constructor is used by tests
	 */
	City(String configFileName) throws IOException {
		Properties props = PropertyLoader.loadProperties(configFileName);
		if (props.getProperty("effectiveProduction")==null){
			throw new IOException("The property file "+configFileName+" does not contain the effectiveProduction property");
		}
		if (props.getProperty("ineffectiveProduction")==null){
			throw new IOException("The property file "+configFileName+" does not contain the ineffectiveProduction property");
		}
		if (props.getProperty("name")==null){
			throw new IOException("The property file "+configFileName+" does not contain the name property");
		}
		if (props.getProperty("kontorType")==null){
			throw new IOException("The property file "+configFileName+" does not contain the kontorType property");
		}
		if (props.getProperty("location.x") == null || props.getProperty("location.y") == null) {
			throw new IOException("The property file "+configFileName+" does not contain the city location");
		}
		if (props.getProperty("effectiveProduction").trim().length()>0){
			String[] wareNames = props.getProperty("effectiveProduction").split(",");
			effectiveProduction = new EWare[wareNames.length];
			for (int i = 0; i < wareNames.length; i++) {
				effectiveProduction[i]=EWare.valueOf(wareNames[i]);
			}
		} else {
			effectiveProduction=null;
		}
		if (props.getProperty("ineffectiveProduction").trim().length()>0){
			String[] wareNames = props.getProperty("ineffectiveProduction").split(",");
			ineffectiveProduction = new EWare[wareNames.length];
			for (int i = 0; i < wareNames.length; i++) {
				ineffectiveProduction[i]=EWare.valueOf(wareNames[i]);
			}
		} else {
			ineffectiveProduction=null;
		}
		String n = props.getProperty("name");
		nameTemplate = n;
		ResourceBundle messages = ResourceBundle.getBundle("ModelMessages", locale.getCurrentLocal());
		name=messages.getString(n);
		kontorType = EKontorType.valueOf(props.getProperty("kontorType"));
		int x = Integer.parseInt(props.getProperty("location.x"));
		int y = Integer.parseInt(props.getProperty("location.y"));
		coordinates = new Point2D(x, y);
		date = new Date(2013);
		initPopulation();
		initWares();
	}
	/**
	 * Init the amount of wares available in the city
	 * This method is protected so it can be overriden by subclasses for testing
	 */
	protected void initWares() {
		for (EWare ware : EWare.values()) {
			boolean hasWare = rnd.nextInt(7)%7!=0;
			if (hasWare){
				int amount = rnd.nextInt(159)+1; // 1..150
				addNewWare(ware, amount);
			} else {
				addNewWare(ware, 0);
			}
		}

	}
	/**
	 * Initialize the population of the different classes based on the properties
	 */
	private void initPopulation() {
		int pop = 1000+(int)Math.abs(rnd.nextDouble()*4000); // value between 1000 and 5000
		int diffPoor = (int) ((rnd.nextDouble()-0.5)*7);
		int poor = (int)((60.0+diffPoor)/100*pop); // about 60% poor
		int begger = calculateInitialBeggarCount(pop);
		setPopulation(begger, EPopulationClass.BEGGAR);
		setPopulation(poor, EPopulationClass.POOR);
		int medium = (pop-poor)*2/3;
		int rich = pop-poor-medium;
		setPopulation(medium, EPopulationClass.MEDIUM);
		setPopulation(rich, EPopulationClass.RICH);

	}

	private int calculateInitialBeggarCount(int totalPopulation) {
		int beerWareAmounts = getWare(EWare.BEER).getAmount();
		int fishWareAmounts = getWare(EWare.FISH).getAmount();
		int grainWareAmounts = getWare(EWare.GRAIN).getAmount();

		int baseAmount = totalPopulation/100;
		double factorAvailablility = 1;
		if (beerWareAmounts == 0) {
			factorAvailablility -= 0.25;
		}
		if (fishWareAmounts == 0) {
			factorAvailablility -= 0.25;
		}
		if (grainWareAmounts == 0) {
			factorAvailablility -= 0.25;
		}
		double amountFactor = 1;
		if (beerWareAmounts > 100) {
			factorAvailablility += 0.25;
		}
		if (fishWareAmounts > 50) {
			factorAvailablility += 0.25;
		}
		if (grainWareAmounts > 20) {
			factorAvailablility += 0.25;
		}
		double randomFactor = rnd.nextDouble() * 2 + 1; // [1,3]
		return (int) (baseAmount * factorAvailablility *amountFactor * randomFactor);

	}

	/**
	 * Count the population.
	 * @return
	 */
	private int internalPopulationCount() {
		int count = 0;
		for (IntegerProperty i : population.values()) {
			count += i.get();
		}
		return count;
	}
	@Override
	public IntegerBinding getPopulationBinding() {
		return totalPopulation;
	}
	/**
	 * Set the population count for a apopulation class
	 * @param population count
	 * @param popClass population class
	 */
	@Override
	public void setPopulation(int population, EPopulationClass popClass) {
		SimpleIntegerProperty populationProperty = new SimpleIntegerProperty(population);
		this.population.put(popClass,populationProperty);
		totalPopulation.bind(populationProperty);
	}
	/**
	 * Retrieve the population count for a class
	 * @param popclass population class
	 * @return
	 */
	@Override
	public int getPopulation(EPopulationClass popclass){
		return population.get(popclass).get();
	}
	@Override
	public IntegerProperty getPopulationProperty(EPopulationClass popclass){
		return population.get(popclass);
	}


	@Override
	public List<IBuilding> getBuildings(){
		return Collections.unmodifiableList(buildings);
	}
	/**
	 * Add a new building to the city
	 * @param building
	 */
	@Override
	public void build(IBuilding building){
		buildings.add(building);
	}
	/**
	 * Remove a building from the list of buildings in the city
	 * @param building
	 */
	@Override
	public void tearDown(IBuilding building){
		buildings.remove(building);
	}
	@Override
	public IReputation getReputation(IPlayer player){
		return reputation.get(player);
	}
	@Override
	public void moveIn(IPlayer player){
		IReputation rep = new Reputation(this,player);
		reputation.put(player, rep);
		playersContributions.put(player, interactionFactory.createContribution());
	}
	@Override
	public ESocialRank getSocialRank(){
		return ESocialRank.CHANDLER; // TODO implement this correctly
	}

	/**
	 * {@inheritDoc}
	 * Update the contributions as the ware is moved
	 */
	@Override
	public int move(IWare ware, int amount,IPlayer player) {
		int moved = super.move(ware, amount, player);
		if (moved<0) {
			if (getWare(ware).getAmount()==0) {
				missingWares.put(ware, date.getCurrentDate());
			}
		} else if (moved>0 && missingWares.containsKey(ware)) {
			missingWares.remove(ware);
		}
		if (player!=null){ // possible from test or from the city itself
			Contributions contrib = playersContributions.get(player);
			contrib.contribute(ware, moved);
		}
		return moved;
	}
	@Override
	public int getContribution(IPlayer player, IWare ware){
		Contributions contribs = playersContributions.get(player);
        if (contribs != null) {
            return contribs.getContribution(ware);
        } else {
            logger.warn("Player "+player+" has no contribution for "+ware);
            // fixme: andi 02/05/14: this hsould not happen
            return 0;
        }
	}
	@Override
	public Map<IWare, DateTime> getMissingWares() {
		return ImmutableMap.copyOf(missingWares);
	}
	@Override
	public List<IPlayer> getResidentPlayers() {
		return ImmutableList.copyOf(reputation.keySet());
	}

    @Override
    public String getUniqueID() {
        return getName();
    }
}