package ch.sahits.game.openpatrician.model.people.impl;

import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.IUpdateableState;
import ch.sahits.game.openpatrician.model.MapSegmentedImage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.CitiesState;
import ch.sahits.game.openpatrician.model.people.IInformant;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.util.CityUtilities;
import ch.sahits.game.openpatrician.util.GenericPair;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import com.google.common.base.Optional;
import lombok.Getter;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Random;

/**
 * Implementation of a informant. Every city can have a informant and is referenced
 * through it's TavernEngine.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 20, 2013
 *
 */
@Prototype
public class InformantState extends BaseTavernPerson implements IInformant, IUpdateableState {

	@Autowired
	private Random rnd;
	@Autowired
	private SeaPiratesState pirateState;
	@Autowired
	private CitiesState cityState;
	@Autowired
	private CityUtilities cityUtils;
	@Autowired
	private IMap map;
	private int price;
    @Getter
	private String name;
    @Getter
	private Optional<GenericPair<ICity, IWare>> wareNeeded = Optional.absent();
    @Getter
    private Optional<GenericPair<ICity, IWare>> wareSurplus = Optional.absent();
    @Getter
    private Optional<GenericPair<String, ICity>> pirateLocation = Optional.absent();
	private boolean showMapIfAvailable;
	/*
	 * Use custom names
	 */
	private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

	public InformantState() {
		name = firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName();
	}


	/**
	 * Update the informant with new pieces of information
	 */
	@Override
	//	@PostConstruct not possible due to initialisation order (on the first city there is no nearest city)
	public void update() {
		int val = rnd.nextInt(2);
		showMapIfAvailable = rnd.nextInt(3) == 0;

		price = rnd.nextInt(1000)+120; // consider age of the game
		updateWareNeedOrSurplus(val);
		int nextVal = rnd.nextInt(2);
		if (val != nextVal) {
			updateWareNeedOrSurplus(nextVal);
		}

		boolean noInformationYet = !wareNeeded.isPresent() && !wareSurplus.isPresent();
		if (rnd.nextBoolean() || noInformationYet) {
			ISeaPirate pirate = pirateState.findSuccessfulPirate();
			ICity nearest = cityUtils.findNearestCity(pirate.getLocation());
			setPirateLocation(pirate, nearest);
		}

	}


	//	/**
	//	 * Set the price of the information
	//	 * @param price
	//	 */
	//	public void setPrice(int price) {
	//		this.price = price;
	//	}

	public int getPrice(IPlayer player) {
		if (showMapIfAvailable && player.getSegmentedMap() != null) {
			return price + rnd.nextInt(1000) + 1000;
		} else {
			return price;
		}

	}

	/**
	 * Set the information about the needed ware in the city.
	 * @param city
	 * @param ware
	 */
	private void setNeededWare(ICity city, IWare ware) {
        wareNeeded = Optional.of(new GenericPair<>(city, ware));
	}



	/**
	 * Set the information about the surplus ware in the city
	 * @param city
	 * @param ware
	 */
	private void setWareSurplus(ICity city, IWare ware) {
        wareSurplus = Optional.of(new GenericPair<>(city, ware));
	}


	/**
	 * Set the location of the pirate.
	 * @param pirate
	 * @param city
	 */
	private void setPirateLocation(ISeaPirate pirate, ICity city) {
		pirateLocation = Optional.of(new GenericPair<>(pirate.getName(), city));
	}

	private void updateWareNeedOrSurplus(int val) {
		switch (val) {
		case 0: // need
			Optional<GenericPair<ICity, IWare>> missing = cityState.findCityWithMostNeededWare();
			if (missing.isPresent()) {
				GenericPair<ICity, IWare> pair = missing.get();
				setNeededWare(pair.getFirst(), pair.getSecond());
			}
			break;
		case 1: // surplus
			Optional<GenericPair<ICity, IWare>> surplus = cityState.findCityWithSurplusWare();
			if (surplus.isPresent()) {
				GenericPair<ICity, IWare> pair = surplus.get();
				setWareSurplus(pair.getFirst(), pair.getSecond());
			}
			break;
		default:
			throw new IllegalStateException("Invalid random number: "+val);
		}
	}

	@Override
	public Optional<MapSegmentedImage> getMapSectionPart(IPlayer player) {
		if (player.getSegmentedMap() != null && showMapIfAvailable){
			MapSegmentedImage segmentedImage = player.getSegmentedMap();
			int index;
			do {
				index = rnd.nextInt(9);
			} while (segmentedImage.getSegmentIndices().contains(index));
			segmentedImage.addSegment(index);
			return Optional.of(segmentedImage);
		} else {
			return Optional.absent();
		}
	}

}
